package ru.yandex.http.proxy;

import java.io.IOException;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import ru.yandex.client.tvm2.ImmutableTvm2ClientConfig;
import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.client.tvm2.Tvm2TicketSupplier;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.util.string.StringUtils;

public class MultiClientTvm2TicketSupplier
    implements Tvm2TicketSupplier, GenericAutoCloseable<IOException>
{
    private static final long DEFAULT_RENEW_INTERVAL = TimeUnit.MINUTES.toMillis(10);
    private final Set<String> destIds;
    private final PrefixedLogger logger;
    private final Tvm2ServiceContextRenewalTask serviceTask;
    private final long renewInterval;

    private Tvm2TicketRenewalTask task;

    public MultiClientTvm2TicketSupplier(
        final Tvm2ServiceContextRenewalTask serviceTask,
        final PrefixedLogger logger,
        final long renewInterval)
    {
        this.logger = logger;
        this.serviceTask = serviceTask;
        this.destIds = new LinkedHashSet<>();
        this.renewInterval = renewInterval;
    }

    public MultiClientTvm2TicketSupplier(
        final PrefixedLogger logger,
        final Tvm2ServiceContextRenewalTask serviceTask)
    {
        this(serviceTask, logger, DEFAULT_RENEW_INTERVAL);
    }

    public synchronized void addDestId(final String destId) {
        this.destIds.add(destId);
    }

    public synchronized void start() {
        if (destIds.size() == 0) {
            logger.info("No tvm2 clients registered, skipping autoapply task");
            return;
        }

        try {
            ImmutableTvm2ClientConfig config =
                new Tvm2ClientConfigBuilder()
                    .destinationClientId(StringUtils.join(destIds, ','))
                    .renewalInterval(renewInterval)
                    .build();
            if (serviceTask == null) {
                throw new IllegalStateException(
                    "Can not start tvm2 client ticket renewal task, no tvm2 service configured");
            }
            task = new Tvm2TicketRenewalTask(logger, serviceTask, config);
            task.start();
        } catch (Exception e) {
            throw new IllegalStateException("Can not start tvm2 client ticket renewal task", e);
        }
    }

    @Override
    public String ticket(final String destId) {
        return task.ticket(destId);
    }

    @Override
    public String ticket() {
        return task.ticket();
    }

    @Override
    public void close() throws IOException {
        if (task != null) {
            task.cancel();
        }
    }
}
