package ru.yandex.http.proxy;

import java.io.IOException;
import java.util.List;
import java.util.function.Supplier;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

public class RequestProxy {
    private final AsyncClient client;

    public RequestProxy(final AsyncClient client) {
        this.client = client;
    }

    protected BasicAsyncRequestProducerGenerator createProducerGenerator(
        final ProxySession session)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestProducerGenerator(
            session.request().getRequestLine().getUri(),
            session.request());
    }

    public void handle(
        final ProxySession session,
        final List<HttpHost> hosts)
        throws HttpException, IOException
    {
        handle(session, hosts, BasicAsyncResponseConsumerFactory.INSTANCE);
    }

    public void handle(
        final ProxySession session,
        final List<HttpHost> hosts,
        final HttpAsyncResponseConsumerFactory<HttpResponse> consumerFactory)
        throws HttpException, IOException
    {
        handle(session, hosts, consumerFactory, new HttpResponseSendingCallback(session));
    }

    @SuppressWarnings("FutureReturnValueIgnored")
    public void handle(
        final ProxySession session,
        final List<HttpHost> hosts,
        final HttpAsyncResponseConsumerFactory<HttpResponse> consumerFactory,
        final FutureCallback<HttpResponse> callback)
        throws HttpException, IOException
    {
        if (hosts.isEmpty()) {
            session.response(HttpStatus.SC_NOT_FOUND);
        } else {
            BasicAsyncRequestProducerGenerator producerGenerator =
                createProducerGenerator(session);

            Header acceptedCharset =
                session.request().getFirstHeader(HttpHeaders.ACCEPT_CHARSET);
            if (acceptedCharset != null) {
                producerGenerator.addHeader(acceptedCharset);
            }

            AsyncClient client =
                this.client.adjustZooHeaders(session.context());

            Supplier<? extends HttpClientContext> contextGenerator =
                session.listener().createContextGeneratorFor(client);

            if (hosts.size() == 1) {
                client.execute(
                    hosts.get(0),
                    producerGenerator,
                    consumerFactory,
                    contextGenerator,
                    callback);
            } else {
                client.execute(
                    hosts,
                    producerGenerator,
                    consumerFactory,
                    contextGenerator,
                    callback);
            }
        }
    }
}

