package ru.yandex.http.util;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.entity.ContentType;

import ru.yandex.charset.Decoder;
import ru.yandex.io.DecodableByteArrayOutputStream;
import ru.yandex.io.NullOutputStream;

public final class CharsetUtils {
    public static final Charset DEFAULT_CHARSET = StandardCharsets.UTF_8;
    public static final ContentType DEFAULT_CONTENT_TYPE =
        ContentType.TEXT_PLAIN.withCharset(DEFAULT_CHARSET);

    private static final byte[] EMPTY_BUF = new byte[0];

    private CharsetUtils() {
    }

    public static ContentType contentType(final HttpEntity entity)
        throws HttpException
    {
        return contentType(entity, DEFAULT_CONTENT_TYPE);
    }

    public static ContentType contentType(
        final HttpEntity entity,
        final ContentType defaultContentType)
        throws HttpException
    {
        try {
            ContentType contentType = null;
            if (entity == null) {
                contentType = defaultContentType;
            } else {
                contentType = ContentType.get(entity);
                if (contentType == null) {
                    contentType = defaultContentType;
                } else if (contentType.getCharset() == null) {
                    contentType = contentType.withCharset(
                        defaultContentType.getCharset());
                }
            }
            return contentType;
        } catch (Throwable t) {
            throw new BadRequestException("Invalid Content-Type header", t);
        }
    }

    public static CharsetDecoder decoder(final HttpEntity entity)
        throws HttpException
    {
        return decoder(entity, DEFAULT_CONTENT_TYPE);
    }

    public static CharsetDecoder decoder(
        final HttpEntity entity,
        final ContentType defaultContentType)
        throws HttpException
    {
        return decoder(contentType(entity, defaultContentType));
    }

    public static CharsetDecoder decoder(final ContentType contentType) {
        return contentType.getCharset()
            .newDecoder()
            .onMalformedInput(CodingErrorAction.REPLACE)
            .onUnmappableCharacter(CodingErrorAction.REPLACE);
    }

    public static Reader content(final HttpEntity entity)
        throws HttpException, IOException
    {
        return content(entity, DEFAULT_CONTENT_TYPE);
    }

    public static Reader content(
        final HttpEntity entity,
        final Charset defaultCharset)
        throws HttpException, IOException
    {
        return content(
            entity,
            DEFAULT_CONTENT_TYPE.withCharset(defaultCharset));
    }

    public static Reader content(
        final HttpEntity entity,
        final ContentType defaultContentType)
        throws HttpException, IOException
    {
        if (entity != null) {
            InputStream content = entity.getContent();
            if (content != null) {
                return new InputStreamReader(
                    content,
                    decoder(entity, defaultContentType));
            }
        }
        return new StringReader("");
    }

    public static Charset acceptedCharset(final HttpRequest request)
        throws HttpException
    {
        return acceptedCharset(request, DEFAULT_CHARSET);
    }

    public static Charset acceptedCharset(
        final HttpRequest request,
        final Charset defaultCharset)
        throws HttpException
    {
        Header header = request.getFirstHeader(HttpHeaders.ACCEPT_CHARSET);
        if (header != null) {
            HeaderElement[] elements = header.getElements();
            if (elements.length > 0) {
                try {
                    return Charset.forName(elements[0].getName());
                } catch (Throwable t) {
                    throw new BadRequestException(
                        "Invalid charset in Accept-Charset", t);
                }
            }
        }
        return defaultCharset;
    }

    public static DecodableByteArrayOutputStream prepareDecodable(
        final HttpEntity entity)
    {
        return prepareDecodable(entity.getContentLength());
    }

    public static DecodableByteArrayOutputStream prepareDecodable(
        final long contentLength)
    {
        if (contentLength == -1L) {
            return new DecodableByteArrayOutputStream();
        } else {
            return new DecodableByteArrayOutputStream((int) contentLength);
        }
    }

    public static DecodableByteArrayOutputStream toDecodable(
        final HttpEntity entity)
        throws IOException
    {
        if (entity != null) {
            DecodableByteArrayOutputStream out = prepareDecodable(entity);
            entity.writeTo(out);
            return out;
        }
        return new DecodableByteArrayOutputStream();
    }

    public static Decoder toDecoder(final HttpEntity entity)
        throws HttpException, IOException
    {
        return toDecoder(entity, CodingErrorAction.REPORT);
    }

    public static Decoder toDecoder(
        final HttpEntity entity,
        final CodingErrorAction errorAction)
        throws HttpException, IOException
    {
        Decoder decoder =
            new Decoder(contentType(entity).getCharset(), errorAction);
        if (entity != null) {
            DecodableByteArrayOutputStream out = toDecodable(entity);
            try {
                out.processWith(decoder);
            } catch (CharacterCodingException e) {
                throw new BadRequestException(
                    "Failed to decode: " + Arrays.toString(out.toByteArray()),
                    e);
            }
        }
        return decoder;
    }

    public static String toString(final HttpEntity entity)
        throws HttpException, IOException
    {
        return toDecoder(entity).toString();
    }

    public static String toString(
        final HttpEntity entity,
        final CodingErrorAction errorAction)
        throws HttpException, IOException
    {
        return toDecoder(entity, errorAction).toString();
    }

    public static byte[] toByteArray(final HttpEntity entity)
        throws IOException
    {
        if (entity != null && entity.getContentLength() != 0L) {
            return toDecodable(entity).toByteArray();
        }
        return EMPTY_BUF;
    }

    public static void consume(final HttpEntity entity) throws IOException {
        if (entity != null) {
            entity.writeTo(NullOutputStream.INSTANCE);
        }
    }
}

