package ru.yandex.http.util;

import java.util.function.Function;
import java.util.function.Supplier;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.ConstFunction;

public class ErrorSuppressingFutureCallback<T> implements FutureCallback<T> {
    public static final Function<Exception, RequestErrorType>
        ALL_ERRORS_AS_HTTP_CLASSIFIER =
            new ConstFunction<>(RequestErrorType.HTTP);

    protected final FutureCallback<? super T> callback;
    protected final Function<Exception, RequestErrorType> errorClassifier;
    protected final Supplier<? extends T> defaultValue;

    public ErrorSuppressingFutureCallback(
        final FutureCallback<? super T> callback,
        final T defaultValue)
    {
        this(callback, () -> defaultValue);
    }

    public ErrorSuppressingFutureCallback(
        final FutureCallback<? super T> callback,
        final Supplier<? extends T> defaultValue)
    {
        this(callback, RequestErrorType.ERROR_CLASSIFIER, defaultValue);
    }

    public ErrorSuppressingFutureCallback(
        final FutureCallback<? super T> callback,
        final Function<Exception, RequestErrorType> errorClassifier,
        final T defaultValue)
    {
        this(callback, errorClassifier, () -> defaultValue);
    }

    public ErrorSuppressingFutureCallback(
        final FutureCallback<? super T> callback,
        final Function<Exception, RequestErrorType> errorClassifier,
        final Supplier<? extends T> defaultValue)
    {
        this.callback = callback;
        this.errorClassifier = errorClassifier;
        this.defaultValue = defaultValue;
    }

    @Override
    public void cancelled() {
        callback.cancelled();
    }

    @Override
    public void failed(final Exception e) {
        switch (errorClassifier.apply(e)) {
            case HOST_NON_RETRIABLE:
            case NON_RETRIABLE:
                callback.failed(e);
                break;
            default:
                completed(defaultValue.get());
                break;
        }
    }

    @Override
    public void completed(final T result) {
        callback.completed(result);
    }
}

