package ru.yandex.http.util;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.concurrent.FutureCallback;

public class FixedMultiFutureCallback<T> {
    private static final long serialVersionUID = 0L;

    private final FutureCallback<? super List<T>> callback;
    private final AtomicInteger requestsLeft;
    private final List<Callback<T>> callbacks;

    public FixedMultiFutureCallback(
        final FutureCallback<? super List<T>> callback,
        final int requestsCount)
    {
        this.callback = callback;
        requestsLeft = new AtomicInteger(requestsCount);
        if (requestsCount == 0) {
            callbacks = null;
            callback.completed(Collections.emptyList());
        } else {
            callbacks = new ArrayList<>(requestsCount);
            for (int i = 0; i < requestsCount; ++i) {
                callbacks.add(new Callback<>());
            }
        }
    }

    public FutureCallback<T> callback(final int i) {
        return callbacks.get(i);
    }

    private void completed() {
        int requestsCount = callbacks.size();
        List<T> result = new ArrayList<>(requestsCount);
        for (int i = 0; i < requestsCount; ++i) {
            result.add(callbacks.get(i).result);
        }
        callback.completed(result);
    }

    public void cancelled() {
        if (requestsLeft.getAndSet(-1) > 0) {
            callback.cancelled();
        }
    }

    public void failed(final Exception e) {
        if (requestsLeft.getAndSet(-1) > 0) {
            callback.failed(e);
        }
    }

    private class Callback<V>
        extends AtomicBoolean
        implements FutureCallback<V>
    {
        private static final long serialVersionUID = 0L;

        private volatile V result = null;

        @Override
        public void cancelled() {
            if (compareAndSet(false, true)) {
                FixedMultiFutureCallback.this.cancelled();
            }
        }

        @Override
        public void failed(final Exception e) {
            if (compareAndSet(false, true)) {
                FixedMultiFutureCallback.this.failed(e);
            }
        }

        @Override
        public void completed(final V result) {
            if (compareAndSet(false, true)) {
                this.result = result;
                if (requestsLeft.decrementAndGet() == 0) {
                    FixedMultiFutureCallback.this.completed();
                }
            }
        }
    }
}

