package ru.yandex.http.util;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;
import java.util.Locale;
import java.util.concurrent.TimeoutException;
import java.util.function.Function;

import org.apache.http.ProtocolException;

public enum RequestErrorType {
    // Something bad happened on network, like SocketTimeoutException,
    // let's try later
    IO,
    // Server responded, but response is not cool, it is some 5xx errors,
    // like 503 Service Unavailable, let's try later
    HTTP,
    // Something very bad happened on network, like UnknownHostException,
    // no more requests should be sent to this host
    HOST_NON_RETRIABLE,
    // Server doesn't like our request, it is 400 Bad Request or something
    // like that. No more requests should be sent with such a bad request
    NON_RETRIABLE;

    public static final Function<Exception, RequestErrorType>
        ERROR_CLASSIFIER =
        new Function<Exception, RequestErrorType>() {
            @Override
            public RequestErrorType apply(final Exception e) {
                RequestErrorType type;
                if (e instanceof IOException) {
                    if (e instanceof SocketTimeoutException) {
                        type = IO;
                    } else if (e instanceof InterruptedIOException) {
                        type = NON_RETRIABLE;
                    } else if (e instanceof UnknownHostException) {
                        type = HOST_NON_RETRIABLE;
                    } else {
                        type = IO;
                    }
                } else if (e instanceof TimeoutException) {
                    type = IO;
                } else if (e instanceof BadResponseException) {
                    int code = ((BadResponseException) e).statusCode();
                    if (ServerErrorStatusPredicate.INSTANCE.test(code)) {
                        type = HTTP;
                    } else if (code == YandexHttpStatus.SC_TOO_MANY_REQUESTS) {
                        type = HOST_NON_RETRIABLE;
                    } else {
                        type = NON_RETRIABLE;
                    }
                } else if (e instanceof ProtocolException) {
                    type = HOST_NON_RETRIABLE;
                } else {
                    type = HTTP;
                }
                return type;
            }
        };

    private final String name =
        name().toLowerCase(Locale.ROOT).replace('_', '-');

    @Override
    public String toString() {
        return name;
    }

    public static boolean temporaryFailure(final Exception e) {
        if (e instanceof ServerException) {
            int code = ((ServerException) e).statusCode();
            switch (code) {
                case YandexHttpStatus.SC_UNAUTHORIZED:
                case YandexHttpStatus.SC_FORBIDDEN:
                case YandexHttpStatus.SC_TOO_MANY_REQUESTS:
                case YandexHttpStatus.SC_BUSY:
                    return true;
                default:
                    break;
            }
        }
        return false;
    }
}

