package ru.yandex.http.util;

import java.util.Set;

import org.apache.http.FormattedHeader;
import org.apache.http.Header;
import org.apache.http.HttpRequest;
import org.apache.http.RequestLine;
import org.apache.http.util.CharArrayBuffer;

public final class RequestToString {
    private RequestToString() {
    }

    private static final int HEADER_OVERHEAD = 4;

    public static StringBuilder requestToStringBuilder(
        final String prefix,
        final HttpRequest request,
        final int suffixLength,
        final Set<String> hiddenHeaders)
    {
        StringBuilder sb;
        if (request == null) {
            sb = new StringBuilder(prefix.length() + 2 + 2 + suffixLength);
            sb.append(prefix);
            sb.append("null");
        } else {
            RequestLine requestLine = request.getRequestLine();
            Header[] headers = request.getAllHeaders();

            sb = new StringBuilder(
                prefix.length()
                + requestStringSize(requestLine, headers)
                + suffixLength);
            sb.append(prefix);
            appendRequest(sb, requestLine, headers, hiddenHeaders);
        }
        return sb;
    }

    public static int requestStringSize(
        final RequestLine requestLine,
        final Header[] headers)
    {
        // XXX: For non-empty headers groups, calculated capacity will be
        // greater than final length by 2, because of overhead introduced
        // by ", " between headers
        HeadersSizeCalculator calc = new HeadersSizeCalculator();
        traverseHeaders(calc, headers);
        return requestLine.getMethod().length()
            + requestLine.getUri().length()
            + calc.size()
            + 2;
    }

    public static void appendRequest(
        final StringBuilder sb,
        final RequestLine requestLine,
        final Header[] headers,
        final Set<String> hiddenHeaders)
    {
        sb.append(requestLine.getMethod());
        sb.append(' ');
        sb.append(requestLine.getUri());
        sb.append(' ');
        traverseHeaders(new HeadersPrinter(sb, hiddenHeaders), headers);
    }

    public interface HeaderVisitor {
        void start();

        void visitHeader(Header header);

        void visitFormattedHeader(FormattedHeader header);

        void finish();
    }

    public static void traverseHeaders(
        final HeaderVisitor visitor,
        final Header[] headers)
    {
        visitor.start();
        for (Header header : headers) {
            if (header instanceof FormattedHeader) {
                visitor.visitFormattedHeader((FormattedHeader) header);
            } else {
                visitor.visitHeader(header);
            }
        }
        visitor.finish();
    }

    public static class HeadersSizeCalculator implements HeaderVisitor {
        private int size = 0;

        @Override
        public void start() {
            ++size; // '['
        }

        @Override
        public void visitHeader(final Header header) {
            size += header.getName().length();
            String value = header.getValue();
            if (value == null) {
                size += 4 + HEADER_OVERHEAD;
            } else {
                size += value.length();
                size += HEADER_OVERHEAD; // ': ' and ', '
            }
        }

        @Override
        public void visitFormattedHeader(final FormattedHeader header) {
            size += header.getBuffer().length() + 2; // ', '
        }

        @Override
        public void finish() {
            ++size; // ']'
        }

        public int size() {
            return size;
        }
    }

    public static class HeadersPrinter implements HeaderVisitor {
        private final StringBuilder sb;
        private final Set<String> hiddenHeaders;
        private final int initialLength;

        public HeadersPrinter(
            final StringBuilder sb,
            final Set<String> hiddenHeaders)
        {
            this.sb = sb;
            this.hiddenHeaders = hiddenHeaders;
            initialLength = sb.length();
        }

        @Override
        public void start() {
            sb.append('[');
        }

        private void appendHiddenHeader(final String name) {
            sb.append(name);
            sb.append(": ..., ");
        }

        @Override
        public void visitHeader(final Header header) {
            String name = header.getName();
            if (hiddenHeaders.contains(name)) {
                appendHiddenHeader(name);
            } else {
                sb.append(name);
                sb.append(':');
                sb.append(' ');
                sb.append(header.getValue());
                sb.append(',');
                sb.append(' ');
            }
        }

        @Override
        public void visitFormattedHeader(final FormattedHeader header) {
            String name = header.getName();
            if (hiddenHeaders.contains(name)) {
                appendHiddenHeader(name);
            } else {
                CharArrayBuffer buffer = header.getBuffer();
                sb.append(buffer.buffer(), 0, buffer.length());
                sb.append(',');
                sb.append(' ');
            }
        }

        @Override
        public void finish() {
            int length = sb.length();
            if (length - initialLength > 1) {
                sb.setLength(length - 2); // ', '
            }
            sb.append(']');
        }
    }
}

