package ru.yandex.http.util;

import java.util.Set;
import java.util.function.BooleanSupplier;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.HttpStatus;
import org.apache.http.HttpVersion;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

public class ResponseConnControl implements HttpResponseInterceptor {
    private static final int MILLIS = 1000;
    private static final Header CONN_CLOSE =
        HeaderUtils.createHeader(HTTP.CONN_DIRECTIVE, HTTP.CONN_CLOSE);
    //Needed for (HTTP < 1.0) explicitly requested keepalive
    private static final Header CONN_KEEPALIVE =
        HeaderUtils.createHeader(HTTP.CONN_DIRECTIVE, HTTP.CONN_CLOSE);

    private final BooleanSupplier keepAlive;
    private final Header timeout;
    private final Set<Integer> keepAliveStatuses;

    public ResponseConnControl(
        final BooleanSupplier keepAlive,
        final int timeout,
        final Set<Integer> keepAliveStatuses)
    {
        this.keepAlive = keepAlive;
        if (timeout < MILLIS << 1) {
            this.timeout = null;
        } else {
            this.timeout = HeaderUtils.createHeader(
                HTTP.CONN_KEEP_ALIVE,
                "timeout=" + Integer.toString(timeout / MILLIS - 1));
        }
        this.keepAliveStatuses = keepAliveStatuses;
    }

    private void addKeepAlive(
        final HttpResponse response,
        final boolean httpRequestIsVeryOldAndIsLessThan1Dot0)
    {
        if (httpRequestIsVeryOldAndIsLessThan1Dot0) {
            response.addHeader(CONN_KEEPALIVE);
        }
        if (timeout != null) {
            Header explicit = response.getFirstHeader(HTTP.CONN_KEEP_ALIVE);
            if (explicit == null) {
                response.addHeader(timeout);
            }
        }
    }

    @Override
    public void process(
        final HttpResponse response,
        final HttpContext context)
    {
        int status = response.getStatusLine().getStatusCode();
        if ((status >= HttpStatus.SC_BAD_REQUEST
                && !keepAliveStatuses.contains(status))
            || !keepAlive.getAsBoolean())
        {
            response.setHeader(CONN_CLOSE);
        } else {
            // Check that connection close is not requested explicitly
            Header explicit = response.getFirstHeader(HTTP.CONN_DIRECTIVE);
            if (explicit == null
                || !HTTP.CONN_CLOSE.equalsIgnoreCase(explicit.getValue()))
            {
                // Check that entity can be properly serialized with keep alive
                HttpEntity entity = response.getEntity();
                if (entity != null
                    && entity.getContentLength() < 0L
                    && (!entity.isChunked()
                        || response.getProtocolVersion()
                            .lessEquals(HttpVersion.HTTP_1_0)))
                {
                    response.setHeader(CONN_CLOSE);
                } else {
                    HttpRequest request = (HttpRequest) context.getAttribute(
                        HttpCoreContext.HTTP_REQUEST);
                    if (request != null) {
                        boolean httpVerLess1Dot0 = request.getProtocolVersion()
                            .lessEquals(HttpVersion.HTTP_1_0);
                        Header requestPreference =
                            request.getFirstHeader(HTTP.CONN_DIRECTIVE);
                        if (requestPreference == null) {
                            if (httpVerLess1Dot0) {
                                response.setHeader(CONN_CLOSE);
                            } else {
                                addKeepAlive(
                                    response,
                                    httpVerLess1Dot0 && explicit == null);
                            }
                        } else if (HTTP.CONN_CLOSE.equalsIgnoreCase(
                            requestPreference.getValue()))
                        {
                            response.setHeader(CONN_CLOSE);
                        } else {
                            addKeepAlive(
                                response,
                                httpVerLess1Dot0 && explicit == null);
                        }
                    }
                }
            }
        }
    }
}

