package ru.yandex.http.util.nio.client;

import java.io.IOException;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import org.apache.http.ConnectionClosedException;
import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.ProtocolException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.NoConnectionReuseStrategy;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.client.HttpAsyncClients;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.concurrent.FailedFuture;
import ru.yandex.function.FunctionBinder;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.BasicFuture;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.http.util.client.DeadlineHttpClientContextSupplier;
import ru.yandex.http.util.client.PlainHttpRoutePlanner;
import ru.yandex.http.util.client.PostsRedirectStrategy;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.pool.AsyncNHttpClientConnectionManager;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.logger.BackendAccessLoggerConfigDefaults;
import ru.yandex.stater.ImmutableStatersConfig;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public abstract class AbstractAsyncClient<C extends AbstractAsyncClient<C>>
    implements GenericAutoCloseable<IOException>
{
    private static final Runnable EMPTY_RUNNABLE = new Runnable() {
        @Override
        public void run() {
        }
    };

    private final AsyncNHttpClientConnectionManager connManager;
    @Nullable
    private final ImmutableStatersConfig statersConfig;
    private final CloseableHttpAsyncClient client;
    private final Supplier<HttpClientContext> httpClientContextGenerator;
    private final ClientContext clientContext;
    private final Charset requestCharset;
    private final boolean passReferer;
    private final RequestContextFactory requestContextFactory;
    private final boolean hostRetries;

    protected AbstractAsyncClient(
        final CloseableHttpAsyncClient client,
        final AbstractAsyncClient<C> sample)
    {
        this.client = client;
        connManager = sample.connManager;
        statersConfig = sample.statersConfig;
        httpClientContextGenerator = sample.httpClientContextGenerator;
        clientContext = sample.clientContext;
        requestCharset = sample.requestCharset;
        passReferer = sample.passReferer;
        requestContextFactory = sample.requestContextFactory;
        hostRetries = sample.hostRetries;
    }

    protected AbstractAsyncClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpTargetConfig backendConfig)
    {
        this(
            reactor,
            backendConfig,
            RequestErrorType.ERROR_CLASSIFIER);
    }

    protected AbstractAsyncClient(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpTargetConfig backendConfig,
        final Function<? super Exception, RequestErrorType> errorClassifier)
    {
        this(
            new AsyncNHttpClientConnectionManager(reactor, backendConfig),
            backendConfig,
            errorClassifier);
    }

    protected AbstractAsyncClient(
        final AsyncNHttpClientConnectionManager connManager,
        final ImmutableHttpTargetConfig backendConfig,
        final Function<? super Exception, RequestErrorType> errorClassifier)
    {
        this.connManager = connManager;
        statersConfig = backendConfig.statersConfig();
        RequestConfig requestConfig = createRequestConfig(backendConfig);
        client =
            new ListeningCloseableHttpAsyncClient(
                createBareClient(connManager, backendConfig, requestConfig));
        httpClientContextGenerator =
            new HttpClientContextGenerator(requestConfig);
        clientContext = new ClientContext(
            errorClassifier,
            connManager.timer(),
            backendConfig);
        requestCharset = backendConfig.requestCharset();
        passReferer = backendConfig.passReferer();
        int sessionTimeout = backendConfig.sessionTimeout();
        if (sessionTimeout > 0) {
            requestContextFactory =
                new SessionTimeoutRequestContextFactory(sessionTimeout);
        } else {
            requestContextFactory = BasicRequestContextFactory.INSTANCE;
        }
        hostRetries = backendConfig.ioRetries().count() != 0
            || backendConfig.httpRetries().count() != 0;
    }

    public C registerClient(
        final AsyncClientRegistrar registrar,
        final String name,
        final ImmutableHttpTargetConfig config)
    {
        return registrar.registerClient(name, adjust(client), config);
    }

    public static RequestConfig createRequestConfig(
        final ImmutableHttpTargetConfig backendConfig)
    {
        return ClientBuilder.createRequestConfig(backendConfig, false);
    }

    public static CloseableHttpAsyncClient createBareClient(
        final AsyncNHttpClientConnectionManager connManager,
        final ImmutableHttpTargetConfig backendConfig)
    {
        return createBareClient(
            connManager,
            backendConfig,
            createRequestConfig(backendConfig));
    }

    public static CloseableHttpAsyncClient createBareClient(
        final AsyncNHttpClientConnectionManager connManager,
        final ImmutableHttpTargetConfig backendConfig,
        final RequestConfig requestConfig)
    {
        HttpAsyncClientBuilder builder = HttpAsyncClients.custom()
            .disableConnectionState()
            .setConnectionManager(connManager)
            .setConnectionManagerShared(true)
            .setDefaultRequestConfig(requestConfig)
            .setHttpProcessor(
                ClientBuilder.prepareHttpProcessorBuilder(backendConfig, false)
                    .add(RequestReporter.INSTANCE)
                    .build())
            .setRoutePlanner(new PlainHttpRoutePlanner(backendConfig.proxy()));
        if (backendConfig.keepAlive()) {
            builder.setConnectionReuseStrategy(
                DefaultConnectionReuseStrategy.INSTANCE);
        } else {
            builder.setConnectionReuseStrategy(
                NoConnectionReuseStrategy.INSTANCE);
        }
        if (backendConfig.redirects() && backendConfig.redirectPosts()) {
            builder.setRedirectStrategy(new PostsRedirectStrategy());
        }
        return builder.build();
    }

    public static boolean serverUnavailalbe(final Throwable t) {
        return t instanceof SocketTimeoutException
            || t instanceof SocketException
            || t instanceof ConnectionClosedException
            || (t instanceof HttpException
                && !(t instanceof ProtocolException));
    }

    public static Header extractReferer(final HttpRequest request) {
        Header referer = request.getFirstHeader(HttpHeaders.REFERER);
        if (referer == null) {
            referer = HeaderUtils.createHeader(
                HttpHeaders.REFERER,
                request.getRequestLine().getUri());
        }
        return referer;
    }

    public C addHeader(final String name, final String value) {
        return addHeader(HeaderUtils.createHeader(name, value));
    }

    public C addHeader(final Header header) {
        return adjust(
            new CustomHeaderCloseableHttpAsyncClient(client, header));
    }

    public C addHeader(final Supplier<List<Header>> supplier) {
        return adjust(
            new CustomDynamicHeadersCloseableHttpClient(client, supplier));
    }

    protected abstract C adjust(final CloseableHttpAsyncClient client);

    public C adjust(final String sessionId, final Header referer) {
        List<Header> headers = new ArrayList<>(2);
        headers.add(
            HeaderUtils.createHeader(
                BackendAccessLoggerConfigDefaults.X_PROXY_SESSION_ID,
                sessionId));
        if (passReferer) {
            headers.add(referer);
        }
        return adjust(
            new CustomHeaderCloseableHttpAsyncClient(client, headers));
    }

    public C adjust(final HttpContext context) {
        String sessionId =
            (String) context.getAttribute(HttpServer.SESSION_ID);
        C result;
        if (passReferer) {
            result = adjust(
                sessionId,
                extractReferer(
                    (HttpRequest) context.getAttribute(
                        HttpCoreContext.HTTP_REQUEST)));
        } else {
            result = addHeader(
                HeaderUtils.createHeader(
                    BackendAccessLoggerConfigDefaults.X_PROXY_SESSION_ID,
                    sessionId));
        }
        if (statersConfig != null) {
            result = result.adjustStater(statersConfig, context);
        }
        return result;
    }

    public C adjustZooHeadersToCheck(final HttpContext context) {
        return adjustZooHeaders(context, true);
    }

    public C adjustZooHeaders(final HttpContext context) {
        return adjustZooHeaders(context, false);
    }

    // Does same same as adjust(HttpContext) + pass zoolooser related headers
    private C adjustZooHeaders(
        final HttpContext context,
        final boolean toCheck)
    {
        HttpRequest request =
            (HttpRequest) context.getAttribute(HttpCoreContext.HTTP_REQUEST);
        List<Header> headers =
            new ArrayList<>(YandexHeaders.ZOO_HEADERS.size() + 2);
        if (passReferer) {
            headers.add(extractReferer(request));
        }
        headers.add(
            HeaderUtils.createHeader(
                BackendAccessLoggerConfigDefaults.X_PROXY_SESSION_ID,
                (String) context.getAttribute(HttpServer.SESSION_ID)));
        CloseableHttpAsyncClient client = this.client;
        for (String name: YandexHeaders.ZOO_HEADERS) {
            Header header = request.getFirstHeader(name);
            if (header != null) {
                if (toCheck
                        && header.getName().equals(YandexHeaders.ZOO_QUEUE_ID))
                {
                    headers.add(
                        HeaderUtils.createHeader(
                            YandexHeaders.ZOO_QUEUE_ID_TO_CHECK,
                            header.getValue()));
                } else {
                    headers.add(header);
                }
            }
        }
        return adjust(
            new CustomHeaderCloseableHttpAsyncClient(client, headers));
    }

    public C adjustStater(
        final ImmutableStatersConfig staters,
        final HttpContext context)
    {
        return adjustStater(
            staters,
            (RequestInfo) context.getAttribute(HttpServer.REQUEST_INFO));
    }

    public C adjustStater(
        final ImmutableStatersConfig staters,
        final RequestInfo request)
    {
        return adjustStater(staters.preparedStaters().get(request));
    }

    public C adjustStater(final RequestsStater stater) {
        return adjust(new StatingCloseableHttpAsyncClient(client, stater));
    }

    public C adjustAsteriskStater() {
        if (statersConfig == null) {
            return adjust(client);
        } else {
            return adjustStater(statersConfig.preparedStaters().asterisk());
        }
    }

    public void start() {
        client.start();
    }

    @Override
    public void close() throws IOException {
        client.close();
        connManager.shutdown();
    }

    public Map<String, Object> status(final boolean verbose) {
        return connManager.status(verbose);
    }

    @Nullable
    public ImmutableStatersConfig statersConfig() {
        return statersConfig;
    }

    public Charset requestCharset() {
        return requestCharset;
    }

    public Supplier<HttpClientContext> httpClientContextGenerator() {
        return httpClientContextGenerator;
    }

    public Future<HttpResponse> execute(
        final Supplier<? extends HttpAsyncRequestProducer> producerSupplier,
        final FutureCallback<? super HttpResponse> callback)
    {
        return execute(
            producerSupplier,
            httpClientContextGenerator(),
            callback);
    }

    public Future<HttpResponse> execute(
        final Supplier<? extends HttpAsyncRequestProducer> producerSupplier,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super HttpResponse> callback)
    {
        return execute(
            producerSupplier,
            BasicAsyncResponseConsumerFactory.ANY_GOOD,
            contextGenerator,
            callback);
    }

    public <T> Future<T> execute(
        final Supplier<? extends HttpAsyncRequestProducer> producerSupplier,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return execute(
            producerSupplier,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public Future<HttpResponse> execute(
        final HttpHost host,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final FutureCallback<? super HttpResponse> callback)
    {
        return execute(
            host,
            producerGenerator,
            httpClientContextGenerator(),
            callback);
    }

    public Future<HttpResponse> execute(
        final HttpHost host,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super HttpResponse> callback)
    {
        return execute(
            new FunctionBinder<>(producerGenerator, host),
            contextGenerator,
            callback);
    }

    public <T> Future<T> execute(
        final Supplier<? extends HttpAsyncRequestProducer> producerSupplier,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        if (hostRetries) {
            return execute(
                new BasicRetryContextFactory(producerSupplier),
                consumerFactory,
                contextGenerator,
                callback);
        } else {
            return requestContextFactory.create(
                client,
                consumerFactory,
                contextGenerator)
                .sendRequest(
                    producerSupplier.get(),
                    contextGenerator.get(),
                    new FilterFutureCallback<>(callback));
        }
    }

    public <T> Future<T> execute(
        final HttpHost host,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return execute(
            host,
            producerGenerator,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public <T> Future<T> execute(
        final HttpHost host,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        return execute(
            new FunctionBinder<>(producerGenerator, host),
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return execute(
            hosts,
            producerGenerator,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        return execute(
            new FailoverRetryContextFactory(hosts, producerGenerator),
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> proxies,
        final Supplier<? extends HttpAsyncRequestProducer> producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return execute(
            proxies,
            producerGenerator,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> proxies,
        final Supplier<? extends HttpAsyncRequestProducer> producerGenerator,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        return execute(
            new MultiProxyRetryContextFactory(proxies, producerGenerator),
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final long deadline,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return execute(
            hosts,
            producerGenerator,
            deadline,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final long deadline,
        final long switchToNextDelay,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return execute(
            hosts,
            producerGenerator,
            deadline,
            switchToNextDelay,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final long deadline,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        long timeLeft = deadline - TimeSource.INSTANCE.currentTimeMillis();
        return execute(
            hosts,
            producerGenerator,
            deadline,
            timeLeft / hosts.size(),
            consumerFactory,
            contextGenerator,
            callback);
    }

    public <T> Future<T> execute(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final long deadline,
        final long switchToNextDelay,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        long timeLeft = deadline - TimeSource.INSTANCE.currentTimeMillis();
        if (timeLeft <= 0L) {
            TimeoutException e = new TimeoutException(
                "Deadline " + deadline + " passed " + -timeLeft + " ms ago");
            callback.failed(e);
            return new FailedFuture<>(e);
        }
        return executeWithDelay(
            hosts,
            new HostToDeadlineRetryContext<>(
                new HostToBasicRetryContext<>(
                    requestContextFactory.create(
                        client,
                        consumerFactory,
                        new DeadlineHttpClientContextSupplier<>(
                            contextGenerator,
                            deadline)),
                    clientContext,
                    producerGenerator),
                deadline),
            switchToNextDelay,
            callback);
    }

    public <T> Future<T> execute(
        final RetryContextFactory retryContextFactory,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        RetryingCallback<T> future =
            new RetryingCallback<>(
                callback,
                retryContextFactory.create(
                    requestContextFactory.create(
                        client,
                        consumerFactory,
                        contextGenerator),
                    clientContext));
        future.sendRequest();
        return future;
    }

    public <T> Future<T> executeWithDelay(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final long delay,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final FutureCallback<? super T> callback)
    {
        return executeWithDelay(
            hosts,
            producerGenerator,
            delay,
            consumerFactory,
            httpClientContextGenerator(),
            callback);
    }

    public <T> Future<T> executeWithDelay(
        final List<HttpHost> hosts,
        final Function<? super HttpHost, ? extends HttpAsyncRequestProducer>
        producerGenerator,
        final long delay,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextGenerator,
        final FutureCallback<? super T> callback)
    {
        return executeWithDelay(
            hosts,
            new HostToBasicRetryContext<>(
                requestContextFactory.create(
                    client,
                    consumerFactory,
                    contextGenerator),
                clientContext,
                producerGenerator),
            delay,
            callback);
    }

    public <T> Future<T> executeWithDelay(
        final List<HttpHost> hosts,
        final Function<HttpHost, RetryContext<T>> retryContextGenerator,
        final long delay,
        final FutureCallback<? super T> callback)
    {
        BasicFuture<T> result = new BasicFuture<>(callback);
        final FutureCallback<T> anyCallback =
            new AnyOfFutureCallback<>(
                result,
                clientContext.errorClassifier(),
                hosts.size());
        Runnable next = EMPTY_RUNNABLE;
        ChainedTaskContext<T> taskContext =
            new ChainedTaskContext<>(retryContextGenerator, result);
        for (int i = hosts.size(); i-- > 0;) {
            HttpHost host = hosts.get(i);
            ChainedTask<T> task =
                new ChainedTask<>(anyCallback, taskContext, host, next);
            clientContext.scheduleRetry(task, delay * i);
            next = task;
        }
        return result;
    }

    public void scheduleRetry(final TimerTask task, final long delay) {
        clientContext.scheduleRetry(task, delay);
    }

    private static class ChainedTaskContext<T>
        implements Function<HttpHost, RetryContext<T>>
    {
        private final Function<HttpHost, RetryContext<T>> contextGenerator;
        // Used for completion checking
        private final Future<T> requestFuture;

        ChainedTaskContext(
            final Function<HttpHost, RetryContext<T>> contextGenerator,
            final Future<T> requestFuture)
        {
            this.contextGenerator = contextGenerator;
            this.requestFuture = requestFuture;
        }

        @Override
        public RetryContext<T> apply(final HttpHost host) {
            return new CompletionCheckingRetryContext<>(
                contextGenerator.apply(host),
                requestFuture);
        }
    }

    private static class ChainedTask<T> extends TimerTask {
        private final AtomicBoolean fired = new AtomicBoolean();
        private final FutureCallback<? super T> callback;
        private final Function<HttpHost, RetryContext<T>> contextProducer;
        private final HttpHost targetHost;
        private final Runnable next;

        ChainedTask(
            final FutureCallback<? super T> callback,
            final Function<HttpHost, RetryContext<T>> contextProducer,
            final HttpHost targetHost,
            final Runnable next)
        {
            this.callback = callback;
            this.contextProducer = contextProducer;
            this.targetHost = targetHost;
            this.next = next;
        }

        @Override
        public void run() {
            if (fired.compareAndSet(false, true)) {
                new ChainedRetryingCallback<>(
                    callback,
                    contextProducer.apply(targetHost),
                    next)
                    .sendRequest();
            } else {
                next.run();
            }
        }
    }

    private static class ChainedRetryingCallback<T>
        extends RetryingCallback<T>
    {
        private final Runnable next;

        ChainedRetryingCallback(
            final FutureCallback<? super T> callback,
            final RetryContext<T> context,
            final Runnable next)
        {
            super(callback, context);
            this.next = next;
        }

        @Override
        public void failed(final Exception e) {
            super.failed(e);
            next.run();
        }
    }
}

