package ru.yandex.http.util.nio.client;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CancellationException;
import java.util.function.BiFunction;

import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.EmptyHttpContext;
import ru.yandex.http.util.HttpHostAppender;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.util.timesource.TimeSource;

public class BasicRequestsListener
    extends ArrayList<RequestInfo>
    implements RequestsListener
{
    private static final long serialVersionUID = 0L;

    private static final int INITIAL_CAPACITY = 10;
    private static final String COMPLETED_IN = " completed in ";
    private static final String FAILED_IN = " failed in ";

    private static final BiFunction<RequestData, Object, RequestInfo>
        HTTP_RESPONSE_CONVERTER =
            (requestData, response) -> new HttpResponseInfo(requestData);

    public BasicRequestsListener() {
        super(INITIAL_CAPACITY);
    }

    public static void appendRoute(
        final StringBuilder sb,
        final HttpRoute route)
    {
        HttpHostAppender.appendTo(sb, route.getTargetHost());
        HttpHost proxy = route.getProxyHost();
        if (proxy != null) {
            sb.append('@');
            HttpHostAppender.appendTo(sb, proxy);
        }
    }

    @Override
    public synchronized long totalTime() {
        int size = size();
        if (size == 0) {
            return 0L;
        } else {
            long startTime = get(0).startTime();
            long endTime = startTime;
            for (int i = 1; i < size; ++i) {
                long time = get(i).endTime();
                if (time > endTime) {
                    endTime = time;
                }
            }
            return endTime - startTime;
        }
    }

    @Override
    public String toString() {
        Map<Object, List<RequestInfo>> groups;
        synchronized (this) {
            int size = size();
            if (size == 0) {
                return "[]";
            } else {
                groups = new LinkedHashMap<>();
            }
            for (int i = 0; i < size; ++i) {
                RequestInfo info = get(i);
                groups.computeIfAbsent(
                    info.requestsGroup(),
                    x -> new ArrayList<>())
                    .add(info);
            }
        }
        StringBuilder sb = new StringBuilder();
        sb.append('[');
        long totalStartTime = Long.MAX_VALUE;
        long totalEndTime = 0L;
        for (List<RequestInfo> group: groups.values()) {
            sb.append('[');
            long groupStartTime = Long.MAX_VALUE;
            long groupEndTime = 0L;
            int size = group.size();
            int i = 0;
            while (true) {
                appendRoute(sb, group.get(i).route());
                if (++i < size) {
                    sb.append(',');
                } else {
                    break;
                }
            }
            sb.append(' ');
            i = 0;
            while (true) {
                RequestInfo request = group.get(i);
                long startTime = request.startTime();
                if (startTime < groupStartTime) {
                    groupStartTime = startTime;
                    if (startTime < totalStartTime) {
                        totalStartTime = startTime;
                    }
                }
                long endTime = request.endTime();
                if (endTime > groupEndTime) {
                    groupEndTime = endTime;
                    if (endTime > totalEndTime) {
                        totalEndTime = endTime;
                    }
                }
                sb.append(endTime - startTime);
                sb.append('(');
                sb.append(request.dnsTime());
                sb.append(',');
                sb.append(request.poolTime());
                sb.append(',');
                sb.append(request.connTime());
                sb.append(')');
                if (++i < size) {
                    sb.append(',');
                } else {
                    break;
                }
            }
            if (size > 1) {
                sb.append('(');
                sb.append(groupEndTime - groupStartTime);
                sb.append(')');
            }
            sb.append(' ');
            i = 0;
            while (true) {
                group.get(i).shortResult(sb);
                if (++i < size) {
                    sb.append(',');
                } else {
                    break;
                }
            }
            sb.append(']');
        }
        if (groups.size() > 1) {
            sb.append('(');
            sb.append(totalEndTime - totalStartTime);
            sb.append(')');
        }
        sb.append(']');
        return new String(sb);
    }

    @Override
    public synchronized String details() {
        int size = size();
        if (size == 0) {
            return "No data";
        } else {
            StringBuilder sb = new StringBuilder("Request execution time: ");
            sb.append(totalTime());
            for (int i = 0; i < size; ++i) {
                sb.append('\n');
                get(i).longResult(sb);
            }
            return new String(sb);
        }
    }

    @Override
    public <T> FutureCallback<? super T> createCallbackFor(
        final HttpRoute route,
        final HttpRequest request,
        final HttpContext context)
    {
        return new Callback<>(
            new RequestData(route, request, context),
            HTTP_RESPONSE_CONVERTER);
    }

    @Override
    public <T> FutureCallback<? super T> createCallbackFor(
        final HttpRoute route,
        final String request,
        final BiFunction<RequestData, ? super T, RequestInfo> responseConverter)
    {
        return new Callback<>(
            new RequestData(route, request, EmptyHttpContext.INSTANCE),
            responseConverter);
    }

    private static class IncompleteRequestInfo extends AbstractRequestInfo {
        IncompleteRequestInfo(final RequestData requestData) {
            super(requestData);
        }

        @Override
        public long endTime() {
            return TimeSource.INSTANCE.currentTimeMillis();
        }

        @Override
        public void shortResult(final StringBuilder sb) {
            sb.append('-');
        }

        @Override
        public void resultDetails(final StringBuilder sb) {
            sb.append(" started ");
            sb.append(TimeSource.INSTANCE.currentTimeMillis() - startTime());
            sb.append(" ms ago (dns ");
            sb.append(dnsTime());
            sb.append(" ms, pool ");
            sb.append(poolTime());
            sb.append(" ms, connect ");
            sb.append(connTime());
            sb.append(" ms) and is not completed yet");
        }
    }

    private static class HttpResponseInfo
        extends AbstractCompletedRequestInfo
    {
        private final HttpResponse response;

        HttpResponseInfo(final RequestData requestData) {
            super(requestData);
            response = requestData.response();
        }

        @Override
        public void shortResult(final StringBuilder sb) {
            if (response == null) {
                sb.append(-1);
            } else {
                sb.append(response.getStatusLine().getStatusCode());
            }
        }

        @Override
        public void resultDetails(final StringBuilder sb) {
            sb.append(COMPLETED_IN);
            writeTiming(sb);
            sb.append(response);
        }
    }

    private static class BadResponseInfo extends AbstractCompletedRequestInfo {
        private final int status;
        private final String response;

        BadResponseInfo(
            final RequestData requestData,
            final BadResponseException e)
        {
            super(requestData);
            status = e.statusCode();
            response = e.response();
        }

        @Override
        public void shortResult(final StringBuilder sb) {
            sb.append(status);
        }

        @Override
        public void resultDetails(final StringBuilder sb) {
            sb.append(FAILED_IN);
            writeTiming(sb);
            sb.append(response);
        }
    }

    private static class ExceptionInfo extends AbstractCompletedRequestInfo {
        private final Exception e;

        ExceptionInfo(
            final RequestData requestData,
            final Exception e)
        {
            super(requestData);
            this.e = e;
        }

        @Override
        public void shortResult(final StringBuilder sb) {
            sb.append(e.getClass().getName());
        }

        @Override
        public void resultDetails(final StringBuilder sb) {
            sb.append(FAILED_IN);
            writeTiming(sb);
            e.printStackTrace(new StringBuilderWriter(sb));
        }
    }

    private class Callback<T> implements FutureCallback<T> {
        private final RequestData requestData;
        private final BiFunction<RequestData, ? super T, RequestInfo>
            responseConverter;
        private final int pos;

        Callback(
            final RequestData requestData,
            final BiFunction<RequestData, ? super T, RequestInfo>
            responseConverter)
        {
            this.requestData = requestData;
            this.responseConverter = responseConverter;
            synchronized (BasicRequestsListener.this) {
                pos = size();
                add(new IncompleteRequestInfo(requestData));
            }
        }

        private void commitResult(final RequestInfo requestInfo) {
            synchronized (BasicRequestsListener.this) {
                set(pos, requestInfo);
            }
        }

        @Override
        public void cancelled() {
            commitResult(
                new ExceptionInfo(
                    requestData,
                    new CancellationException()));
        }

        @Override
        public void completed(final T result) {
            commitResult(responseConverter.apply(requestData, result));
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof BadResponseException) {
                commitResult(
                    new BadResponseInfo(
                        requestData,
                        (BadResponseException) e));
            } else {
                commitResult(new ExceptionInfo(requestData, e));
            }
        }
    }
}

