package ru.yandex.http.util.nio.client;

import java.util.ArrayList;
import java.util.List;
import java.util.TimerTask;
import java.util.concurrent.Future;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.util.timesource.TimeSource;

public abstract class MultiTargetRetryContextBase<T>
    implements RetryContext<T>
{
    protected final RequestContext<T> requestContext;
    private final ClientContext clientContext;
    private final List<HostInfo> hosts;
    private int host = 0;

    protected MultiTargetRetryContextBase(
        final MultiTargetRequestContext<T> context)
    {
        requestContext = context.requestContext();
        clientContext = context.clientContext();
        List<HttpHost> hosts = context.hosts();
        this.hosts = new ArrayList<>(hosts.size());
        int ioRetries = clientContext.ioRetries().count();
        int httpRetries = clientContext.httpRetries().count();
        for (HttpHost host: hosts) {
            this.hosts.add(new HostInfo(host, ioRetries, httpRetries));
        }
    }

    protected abstract Future<T> sendRequest(
        final HttpHost host,
        final FutureCallback<T> callback);

    @Override
    public Future<T> sendRequest(final FutureCallback<T> callback) {
        return sendRequest(hosts.get(host).host(), callback);
    }

    @Override
    public long nextRetryInterval(final Exception e) {
        HostInfo host = hosts.get(this.host);
        long now = TimeSource.INSTANCE.currentTimeMillis();
        switch (clientContext.detectErrorType(e)) {
            case IO:
                host.decIoRetries();
                host.nextRetryAt(now + clientContext.ioRetries().interval());
                break;
            case HTTP:
                host.decHttpRetries();
                host.nextRetryAt(now + clientContext.httpRetries().interval());
                break;
            case HOST_NON_RETRIABLE:
                host.disableHost();
                break;
            default:
                return -1L;
        }

        long minNextRetryAt = Long.MAX_VALUE;
        int minHost = -1;
        for (int i = 1; i <= hosts.size(); ++i) {
            int pos = (this.host + i) % hosts.size();
            host = hosts.get(pos);
            long nextRetryAt = host.nextRetryAt();
            if (nextRetryAt < minNextRetryAt && host.retriesLeft() >= 0) {
                minNextRetryAt = nextRetryAt;
                minHost = pos;
            }
        }
        long interval;
        if (minHost == -1) {
            interval = -1L;
        } else {
            this.host = minHost;
            interval = Math.max(0, minNextRetryAt - now);
        }
        return interval;
    }

    @Override
    public void scheduleRetry(final TimerTask task, final long delay) {
        clientContext.scheduleRetry(task, delay);
    }

    private static class HostInfo {
        private final HttpHost host;
        private long nextRetryAt = 0L;
        private boolean lastErrorWasIO = false;
        private int ioRetries;
        private int httpRetries;

        HostInfo(
            final HttpHost host,
            final int ioRetries,
            final int httpRetries)
        {
            this.host = host;
            this.ioRetries = ioRetries;
            this.httpRetries = httpRetries;
        }

        public HttpHost host() {
            return host;
        }

        public long nextRetryAt() {
            return nextRetryAt;
        }

        public void nextRetryAt(final long nextRetryAt) {
            this.nextRetryAt = nextRetryAt;
        }

        public void disableHost() {
            ioRetries = -1;
            httpRetries = -1;
        }

        public int retriesLeft() {
            if (lastErrorWasIO) {
                return ioRetries;
            } else {
                return httpRetries;
            }
        }

        public void decIoRetries() {
            lastErrorWasIO = true;
            --ioRetries;
        }

        public void decHttpRetries() {
            lastErrorWasIO = false;
            --httpRetries;
        }
    }
}

