package ru.yandex.http.util.nio.client;

import java.util.TimerTask;
import java.util.concurrent.Future;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.BasicFuture;

public class RetryingCallback<T> extends BasicFuture<T> {
    private final RetryContext<T> context;
    private Future<T> subrequest = null;

    public RetryingCallback(
        final FutureCallback<? super T> callback,
        final RetryContext<T> context)
    {
        super(callback);
        this.context = context;
    }

    public synchronized void sendRequest() {
        subrequest = context.sendRequest(this);
    }

    @Override
    protected boolean onCancel(final boolean mayInterruptIfRunning) {
        boolean result = super.onCancel(mayInterruptIfRunning);
        Future<T> subrequest;
        synchronized (this) {
            subrequest = this.subrequest;
            this.subrequest = null;
        }
        if (subrequest != null
            && !subrequest.cancel(mayInterruptIfRunning)
            && !subrequest.isCancelled())
        {
            result = false;
        }
        return result;
    }

    @Override
    public void failed(final Exception e) {
        synchronized (this) {
            if (done) {
                return;
            }
            long interval = context.nextRetryInterval(e);
            if (interval == -1L) {
                done = true;
            } else {
                subrequest = null;
                context.scheduleRetry(new RetryTask(), interval);
                return;
            }
        }
        this.e = e;
        onFailure(e);
        synchronized (this) {
            completed = true;
            notifyAll();
        }
    }

    private class RetryTask extends TimerTask {
        @Override
        public void run() {
            synchronized (RetryingCallback.this) {
                if (!done) {
                    subrequest = context.sendRequest(RetryingCallback.this);
                }
            }
        }
    }
}

