package ru.yandex.http.util.nio.client;

import java.io.IOException;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;

import org.apache.http.HttpHost;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.reactor.IOSession;
import org.apache.http.nio.reactor.ssl.SSLIOSession;
import org.apache.http.nio.reactor.ssl.SSLMode;
import org.apache.http.nio.reactor.ssl.SSLSetupHandler;

import ru.yandex.http.config.ImmutableClientHttpsConfig;
import ru.yandex.http.util.client.ClientBuilder;

public class SSLIOSessionStrategy implements SchemeIOSessionStrategy {
    private final ImmutableClientHttpsConfig config;
    private final SSLContext sslContext;
    private final HostnameVerifier hostnameVerifier;

    public SSLIOSessionStrategy(final ImmutableClientHttpsConfig config) {
        this.config = config;
        sslContext = config.getSSLContext();
        hostnameVerifier = ClientBuilder.createHostnameVerifier(config);
    }

    @Override
    public boolean isLayeringRequired() {
        return true;
    }

    @Override
    public SSLIOSession upgrade(final HttpHost host, final IOSession session)
        throws IOException
    {
        SSLIOSession sslSession = new SSLIOSession(
            session,
            SSLMode.CLIENT,
            host,
            sslContext,
            new SetupHandler(config, hostnameVerifier, host));
        session.setAttribute(SSLIOSession.SESSION_KEY, sslSession);
        sslSession.initialize();
        return sslSession;
    }

    private static class SetupHandler implements SSLSetupHandler {
        private final ImmutableClientHttpsConfig config;
        private final HostnameVerifier hostnameVerifier;
        private final HttpHost host;

        SetupHandler(
            final ImmutableClientHttpsConfig config,
            final HostnameVerifier hostnameVerifier,
            final HttpHost host)
        {
            this.config = config;
            this.hostnameVerifier = hostnameVerifier;
            this.host = host;
        }

        @Override
        public void initalize(final SSLEngine sslEngine) {
            config.initialize(sslEngine);
        }

        @Override
        public void verify(
            final IOSession ioSession,
            final SSLSession sslSession)
            throws SSLException
        {
            String hostname = host.getHostName();
            if (!hostnameVerifier.verify(hostname, sslSession)) {
                Certificate[] certs = sslSession.getPeerCertificates();
                X509Certificate cert = (X509Certificate) certs[0];
                throw new SSLPeerUnverifiedException(
                    "Host name '" + hostname
                    + "' doesn't match server certificate: "
                    + cert.getSubjectX500Principal());
            }
        }
    }
}

