package ru.yandex.http.util.nio.client;

import java.io.IOException;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.NHttpConnection;
import org.apache.http.nio.client.HttpAsyncClient;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.reactor.IOEventDispatch;

import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.LazyAsyncResponseConsumer;
import ru.yandex.util.timesource.TimeSource;

public class SessionTimeoutRequestContext<T> extends RequestContext<T> {
    private final int timeout;

    // CSOFF: ParameterNumber
    public SessionTimeoutRequestContext(
        final HttpAsyncClient client,
        final HttpAsyncResponseConsumerFactory<T> consumerFactory,
        final Supplier<? extends HttpClientContext> contextSupplier,
        final int timeout)
    {
        super(client, consumerFactory, contextSupplier);

        this.timeout = timeout;
    }
    // CSON: ParameterNumber

    @Override
    public Future<T> sendRequest(
        final HttpAsyncRequestProducer requestProducer,
        final HttpClientContext context,
        final FutureCallback<T> callback)
    {
        return client.execute(
            requestProducer,
            new SessionAsyncResponseConsumer<>(
                consumerFactory,
                requestProducer,
                context,
                timeout),
            context,
            callback);
    }

    private static class SessionAsyncResponseConsumer<T>
        extends LazyAsyncResponseConsumer<T>
    {
        private final HttpClientContext context;
        private final int timeout;
        private final long startTime;

        SessionAsyncResponseConsumer(
            final HttpAsyncResponseConsumerFactory<T> factory,
            final HttpAsyncRequestProducer producer,
            final HttpClientContext context,
            final int timeout)
        {
            super(factory, producer);

            this.context = context;
            this.timeout = timeout;

            startTime = TimeSource.INSTANCE.currentTimeMillis();
        }

        @Override
        public void responseReceived(final HttpResponse response)
            throws HttpException, IOException
        {
            long now = TimeSource.INSTANCE.currentTimeMillis();
            int left = (int) (timeout - now + startTime);
            if (left > 0) {
                NHttpConnection conn =
                    (NHttpConnection) context.getAttribute(
                        IOEventDispatch.CONNECTION_KEY);

                conn.setSocketTimeout(left);
            }

            super.responseReceived(response);
        }
    }
}
