package ru.yandex.http.util.nio.client;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.net.InetSocketAddress;
import java.nio.channels.CancelledKeyException;
import java.util.Map;
import java.util.Timer;
import java.util.logging.Level;

import org.apache.http.conn.DnsResolver;
import org.apache.http.impl.nio.DefaultNHttpClientConnection;
import org.apache.http.impl.nio.reactor.AbstractIODispatch;
import org.apache.http.impl.nio.reactor.DefaultConnectingIOReactor;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.apache.http.nio.NHttpClientEventHandler;
import org.apache.http.nio.reactor.IOReactorException;
import org.apache.http.nio.reactor.IOSession;
import org.apache.http.nio.reactor.SessionRequest;
import org.apache.http.nio.reactor.SessionRequestCallback;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.concurrent.SingleNamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.http.util.nio.LoggingIOReactorExceptionHandler;
import ru.yandex.http.util.nio.ReactorUtils;
import ru.yandex.http.util.nio.client.pool.AsyncDnsResolver;
import ru.yandex.http.util.nio.client.pool.DnsUpdater;
import ru.yandex.http.util.nio.client.pool.MultiDnsResolver;
import ru.yandex.http.util.nio.client.pool.SingleThreadDnsResolverFactory;
import ru.yandex.http.util.nio.client.pool.SingleThreadDnsUpdater;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.logger.PrefixedLogger;

public class SharedConnectingIOReactor
    extends AbstractIODispatch<DefaultNHttpClientConnection>
    implements GenericAutoCloseable<IOException>, Runnable
{
    private final AsyncDnsResolver dnsResolver;
    private final DnsUpdater dnsUpdater;
    private final PrefixedLogger logger;
    private final DefaultConnectingIOReactor reactor;
    private final NHttpClientEventHandler handler;
    private final Timer timer;
    private final Thread thread;

    public SharedConnectingIOReactor(
        final ImmutableBaseServerConfig serverConfig,
        final ImmutableDnsConfig dnsConfig)
        throws IOReactorException
    {
        this(
            serverConfig,
            dnsConfig,
            new ThreadGroup(serverConfig.name() + "-C"));
    }

    public SharedConnectingIOReactor(
        final ImmutableBaseServerConfig serverConfig,
        final ImmutableDnsConfig dnsConfig,
        final ThreadGroup threadGroup)
        throws IOReactorException
    {
        this(
            serverConfig,
            dnsConfig,
            ClientBuilder.createDnsResolver(dnsConfig),
            threadGroup);
    }

    // CSOFF: ParameterNumber
    public SharedConnectingIOReactor(
        final ImmutableBaseServerConfig serverConfig,
        final ImmutableDnsConfig dnsConfig,
        final DnsResolver dnsResolver,
        final ThreadGroup threadGroup)
        throws IOReactorException
    {
        this(
            new MultiDnsResolver(
                serverConfig.workers(),
                new SingleThreadDnsResolverFactory(
                    dnsResolver,
                    dnsConfig.dnsTtl(),
                    dnsConfig.dnsUpdateInterval(),
                    new NamedThreadFactory(
                        threadGroup,
                        threadGroup.getName() + "-DNS-",
                        true))),
            new SingleThreadDnsUpdater(
                dnsResolver,
                dnsConfig.dnsUpdateInterval(),
                new SingleNamedThreadFactory(
                    threadGroup,
                    threadGroup.getName() + "-DNS-Updater",
                    true)),
            serverConfig,
            threadGroup);
    }

    public SharedConnectingIOReactor(
        final AsyncDnsResolver dnsResolver,
        final DnsUpdater dnsUpdater,
        final ImmutableBaseServerConfig serverConfig,
        final ThreadGroup threadGroup)
        throws IOReactorException
    {
        this.dnsResolver = dnsResolver;
        this.dnsUpdater = dnsUpdater;
        logger = serverConfig.loggers().preparedLoggers().asterisk()
            .addPrefix(threadGroup.getName());
        reactor =
            new DefaultConnectingIOReactor(
                IOReactorConfig.custom()
                    .setIoThreadCount(serverConfig.workers())
                    .setSelectInterval(serverConfig.timerResolution())
                    .setSoTimeout(0)
                    .setConnectTimeout(0)
                    .setSoReuseAddress(true)
                    .setSoKeepAlive(true)
                    .setTcpNoDelay(true)
                    .setSoLinger(0)
                    .build(),
                new NamedThreadFactory(
                    threadGroup,
                    threadGroup.getName() + '-',
                    true));
        reactor.setExceptionHandler(
            new LoggingIOReactorExceptionHandler(logger));
        handler = new LoggingAsyncRequestExecutor(logger);
        timer = new Timer(threadGroup.getName() + "-RetryTimer", true);
        thread =
            new Thread(threadGroup, this, threadGroup.getName() + "-Reactor");
        thread.setDaemon(true);
    }

    public AsyncDnsResolver dnsResolver() {
        return dnsResolver;
    }

    public DnsUpdater dnsUpdater() {
        return dnsUpdater;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public Timer timer() {
        return timer;
    }

    public SessionRequest connect(
        final InetSocketAddress remoteAddress,
        final InetSocketAddress localAddress,
        final SessionRequestCallback callback)
    {
        return reactor.connect(remoteAddress, localAddress, null, callback);
    }

    public void status(
        final Map<String, Object> status,
        final boolean verbose)
    {
        ReactorUtils.status(status, reactor);
    }

    public void start() {
        dnsResolver.start();
        dnsUpdater.start();
        thread.start();
    }

    @Override
    public void run() {
        try {
            reactor.execute(this);
        } catch (InterruptedIOException | IOReactorException e) {
            logger.log(Level.SEVERE, "Reactor died", e);
        } catch (Throwable e) {
            logger.log(Level.SEVERE, "Reactor BAD exception", e);
            throw e;
        }
    }

    @Override
    @SuppressWarnings("ThreadJoinLoop")
    public void close() throws IOException {
        dnsUpdater.close();
        dnsResolver.close();
        reactor.shutdown();
        timer.cancel();
        try {
            thread.join();
        } catch (InterruptedException e) {
        }
    }

    // AbstractIODispatch implementation
    @Override
    protected DefaultNHttpClientConnection createConnection(
        final IOSession session)
    {
        logger.warning("SharedReactor createConnection");
        throw new CancelledKeyException();
    }

    @Override
    protected void onConnected(final DefaultNHttpClientConnection conn) {
        try {
            handler.connected(
                conn,
                conn.getContext().getAttribute(IOSession.ATTACHMENT_KEY));
        } catch (Exception e) {
            handler.exception(conn, e);
        }
    }

    @Override
    protected void onClosed(final DefaultNHttpClientConnection conn) {
        handler.closed(conn);
    }

    @Override
    protected void onException(
        final DefaultNHttpClientConnection conn,
        final IOException e)
    {
        handler.exception(conn, e);
    }

    @Override
    protected void onInputReady(final DefaultNHttpClientConnection conn) {
        conn.consumeInput(handler);
    }

    @Override
    protected void onOutputReady(final DefaultNHttpClientConnection conn) {
        conn.produceOutput(handler);
    }

    @Override
    protected void onTimeout(final DefaultNHttpClientConnection conn) {
        try {
            handler.timeout(conn);
        } catch (Exception e) {
            handler.exception(conn, e);
        }
    }
}

