package ru.yandex.http.util.nio.client;

import java.io.IOException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeoutException;

import org.apache.http.HttpResponse;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.nio.protocol.HttpAsyncResponseConsumer;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.DuplexFutureCallback;
import ru.yandex.http.util.IdempotentFutureCallback;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.ContentDecoderListener;
import ru.yandex.http.util.nio.ContentEncoderListener;
import ru.yandex.http.util.nio.CountingAsyncRequestProducer;
import ru.yandex.http.util.nio.CountingAsyncResponseConsumer;
import ru.yandex.stater.RequestInfo;
import ru.yandex.stater.RequestsStater;
import ru.yandex.util.timesource.TimeSource;

public class StatingCloseableHttpAsyncClient
    extends FilterCloseableHttpAsyncClient
{
    private final RequestsStater stater;

    public StatingCloseableHttpAsyncClient(
        final CloseableHttpAsyncClient client,
        final RequestsStater stater)
    {
        super(client);
        this.stater = stater;
    }

    @Override
    public <T> Future<T> execute(
        final HttpAsyncRequestProducer requestProducer,
        final HttpAsyncResponseConsumer<T> responseConsumer,
        final HttpContext context,
        final FutureCallback<T> callback)
    {
        StatingCallback statingCallback =
            new StatingCallback(stater, context);
        return super.execute(
            new CountingAsyncRequestProducer(requestProducer, statingCallback),
            new CountingAsyncResponseConsumer<>(
                responseConsumer,
                statingCallback),
            context,
            new IdempotentFutureCallback<>(
                new DuplexFutureCallback<>(
                    statingCallback,
                    callback)));
    }

    private static class StatingCallback
        implements ContentDecoderListener,
            ContentEncoderListener,
            FutureCallback<Object>
    {
        private final long start = TimeSource.INSTANCE.currentTimeMillis();
        private final RequestsStater stater;
        private final HttpContext context;
        private long bytesRead = 0L;
        private long bytesWritten = 0L;

        StatingCallback(
            final RequestsStater stater,
            final HttpContext context)
        {
            this.stater = stater;
            this.context = context;
        }

        private void stat(final int status) {
            stater.accept(
                new RequestInfo(
                    TimeSource.INSTANCE.currentTimeMillis(),
                    status,
                    start,
                    start,
                    bytesWritten,
                    bytesRead));
        }

        @Override
        public void cancelled() {
            stat(YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
        }

        @Override
        public void completed(final Object result) {
            HttpResponse response = (HttpResponse)
                context.getAttribute(HttpCoreContext.HTTP_RESPONSE);
            int statusCode;
            if (response == null) {
                statusCode = YandexHttpStatus.SC_OK;
            } else {
                statusCode = response.getStatusLine().getStatusCode();
            }
            stat(statusCode);
        }

        @Override
        public void failed(final Exception e) {
            int statusCode;
            if (e instanceof BadResponseException) {
                statusCode = ((BadResponseException) e).statusCode();
            } else if (e instanceof IOException) {
                statusCode = YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST;
            } else if (e instanceof TimeoutException) {
                statusCode = YandexHttpStatus.SC_BUSY;
            } else {
                statusCode = YandexHttpStatus.SC_INTERNAL_SERVER_ERROR;
            }
            stat(statusCode);
        }

        @Override
        public void incrementRead(final long read) {
            bytesRead += read;
        }

        @Override
        public void onDecoderIOException(final IOException e) {
        }

        @Override
        public void onDecoderError(final Throwable t) {
        }

        @Override
        public void incrementWritten(final long written) {
            bytesWritten += written;
        }

        @Override
        public void onEncoderIOException(final IOException e) {
        }

        @Override
        public void onEncoderError(final Throwable t) {
        }
    }
}

