package ru.yandex.http.util.nio.client.pool;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.nio.NHttpClientConnection;
import org.apache.http.nio.conn.NHttpClientConnectionManager;
import org.apache.http.nio.conn.NoopIOSessionStrategy;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.reactor.IOEventDispatch;
import org.apache.http.nio.reactor.IOSession;
import org.apache.http.nio.reactor.SessionRequest;
import org.apache.http.nio.reactor.SessionRequestCallback;
import org.apache.http.pool.ConnPoolControl;
import org.apache.http.pool.PoolStats;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.BasicFuture;
import ru.yandex.http.util.nio.client.SSLIOSessionStrategy;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.util.timesource.TimeSource;

public class AsyncNHttpClientConnectionManager
    implements NHttpClientConnectionManager, ConnPoolControl<HttpRoute>
{
    private final ConcurrentHashMap<HttpRoute, RoutePool> routes =
        new ConcurrentHashMap<>();
    private final AtomicInteger pendingPools = new AtomicInteger();
    private final SharedConnectingIOReactor reactor;
    private final ConnectionConfig connectionConfig;
    private final SSLIOSessionStrategy sslIOSessionStrategy;
    private volatile int maxConnectionsPerRoute;

    public AsyncNHttpClientConnectionManager(
        final SharedConnectingIOReactor reactor,
        final ImmutableHttpTargetConfig backendConfig)
    {
        this.reactor = reactor;
        connectionConfig = backendConfig.toConnectionConfig();
        sslIOSessionStrategy =
            new SSLIOSessionStrategy(backendConfig.httpsConfig());
        maxConnectionsPerRoute = backendConfig.connections();
    }
    // CSON: ParameterNumber

    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        reactor.status(status, verbose);
        status.put("pending_pools", pendingPools.get());
        if (verbose) {
            List<RouteWithStats> routes = new ArrayList<>();
            this.routes.forEach(
                (route, pool)
                    -> routes.add(new RouteWithStats(route, pool.stats())));
            Collections.sort(routes, Collections.reverseOrder());
            for (RouteWithStats route: routes) {
                PoolStats stats = route.stats();
                if (stats.getLeased() == 0
                    && stats.getPending() == 0
                    && stats.getAvailable() == 0)
                {
                    break;
                }
                Map<String, Object> routeStatus = new LinkedHashMap<>();
                routeStatus.put("active_connections", stats.getLeased());
                routeStatus.put("pending_connections", stats.getPending());
                routeStatus.put("available_connections", stats.getAvailable());
                status.put(route.route(), routeStatus);
            }
        }
        return status;
    }

    public PrefixedLogger logger() {
        return reactor.logger();
    }

    public SessionRequest connect(
        final InetSocketAddress remoteAddress,
        final InetSocketAddress localAddress,
        final SessionRequestCallback callback)
    {
        return reactor.connect(remoteAddress, localAddress, callback);
    }

    public Timer timer() {
        return reactor.timer();
    }

    public PooledNHttpClientConnection createConnection(
        final RoutePool routePool,
        final IOSession session)
    {
        PooledNHttpClientConnection conn = new PooledNHttpClientConnection(
            session,
            connectionConfig,
            routePool);
        session.setAttribute(IOEventDispatch.CONNECTION_KEY, conn);
        return conn;
    }

    // NHttpClientConnectionManager implementation
    @Override
    public void execute(final IOEventDispatch eventDispatch) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void shutdown() throws IOException {
        for (RoutePool pool: routes.values()) {
            pool.close();
        }
    }

    // CSOFF: ParameterNumber
    @Override
    @SuppressWarnings("FutureReturnValueIgnored")
    public Future<NHttpClientConnection> requestConnection(
        final HttpRoute route,
        final Object state,
        final long connectTimeout,
        final long connectionRequestTimeout,
        final TimeUnit timeUnit,
        final FutureCallback<NHttpClientConnection> callback)
    {
        RoutePool pool = routes.get(route);
        if (pool == null) {
            PoolRequest request = new PoolRequest(
                route,
                timeUnit.toMillis(connectTimeout),
                timeUnit.toMillis(connectionRequestTimeout),
                callback);
            reactor.dnsResolver().resolve(
                targetHost(route).getHostName(),
                request.dnsCallback());
            pendingPools.incrementAndGet();
            return request;
        } else {
            return pool.requestConnection(
                timeUnit.toMillis(connectTimeout),
                timeUnit.toMillis(connectionRequestTimeout),
                callback);
        }
    }

    @Override
    public void releaseConnection(
        final NHttpClientConnection conn,
        final Object newState,
        final long validDuration,
        final TimeUnit timeUnit)
    {
        if (conn instanceof TimedConnection) {
            ((TimedConnection) conn).conn()
                .releaseConnection(timeUnit.toMillis(validDuration));
        }
    }
    // CSON: ParameterNumber

    @Override
    public void startRoute(
        final NHttpClientConnection conn,
        final HttpRoute route,
        final HttpContext context)
        throws IOException
    {
        if (conn instanceof TimedConnection) {
            ((TimedConnection) conn).conn().startRoute(targetHost(route));
        }
    }

    @Override
    public void upgrade(
        final NHttpClientConnection conn,
        final HttpRoute route,
        final HttpContext context)
        throws IOException
    {
        if (conn instanceof TimedConnection) {
            ((TimedConnection) conn).conn().startRoute(route.getTargetHost());
        }
    }

    @Override
    public void routeComplete(
        final NHttpClientConnection conn,
        final HttpRoute route,
        final HttpContext context)
    {
        if (conn instanceof TimedConnection) {
            ((TimedConnection) conn).conn().markRouteComplete();
        }
    }

    @Override
    public boolean isRouteComplete(final NHttpClientConnection conn) {
        if (conn instanceof TimedConnection) {
            return ((TimedConnection) conn).conn().isRouteComplete();
        }
        return false;
    }

    @Override
    public void closeIdleConnections(
        final long idleTimeout,
        final TimeUnit timeUnit)
    {
        long now = TimeSource.INSTANCE.currentTimeMillis();
        long idleSince = now - timeUnit.toMillis(idleTimeout);
        for (RoutePool pool: routes.values()) {
            pool.closeIdleConnections(idleSince);
        }
    }

    @Override
    public void closeExpiredConnections() {
        long now = TimeSource.INSTANCE.currentTimeMillis();
        for (RoutePool pool: routes.values()) {
            pool.closeExpiredConnections(now);
        }
    }

    // ConnPoolControl implementation
    @Override
    public void setMaxTotal(final int max) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int getMaxTotal() {
        return Integer.MAX_VALUE;
    }

    @Override
    public void setDefaultMaxPerRoute(final int max) {
        maxConnectionsPerRoute = max;
    }

    @Override
    public int getDefaultMaxPerRoute() {
        return maxConnectionsPerRoute;
    }

    @Override
    public void setMaxPerRoute(final HttpRoute route, final int max) {
        throw new UnsupportedOperationException();
    }

    @Override
    public int getMaxPerRoute(final HttpRoute route) {
        return maxConnectionsPerRoute;
    }

    @Override
    public PoolStats getTotalStats() {
        int leased = 0;
        int pending = 0;
        int available = 0;
        for (RoutePool pool: routes.values()) {
            PoolStats stats = pool.stats();
            leased += stats.getLeased();
            pending += stats.getPending();
            available += stats.getAvailable();
        }
        return new PoolStats(leased, pending, available, getMaxTotal());
    }

    @Override
    public PoolStats getStats(final HttpRoute route) {
        RoutePool pool = routes.get(route);
        if (pool == null) {
            return new PoolStats(0, 0, 0, maxConnectionsPerRoute);
        } else {
            return pool.stats();
        }
    }

    public static HttpHost targetHost(final HttpRoute route) {
        HttpHost proxyHost = route.getProxyHost();
        if (proxyHost == null) {
            return route.getTargetHost();
        } else {
            return proxyHost;
        }
    }

    public SchemeIOSessionStrategy selectIOSessionStrategy(
        final HttpHost targetHost)
    {
        if ("https".equalsIgnoreCase(targetHost.getSchemeName())) {
            return sslIOSessionStrategy;
        } else {
            return NoopIOSessionStrategy.INSTANCE;
        }
    }

    private static class RouteWithStats implements Comparable<RouteWithStats> {
        private final String route;
        private final PoolStats stats;

        RouteWithStats(final HttpRoute route, final PoolStats stats) {
            this.route = route.toString();
            this.stats = stats;
        }

        public String route() {
            return route;
        }

        public PoolStats stats() {
            return stats;
        }

        @Override
        public int compareTo(final RouteWithStats other) {
            int cmp = Integer.compare(
                stats.getLeased(),
                other.stats.getLeased());
            if (cmp == 0) {
                cmp = Integer.compare(
                    stats.getPending(),
                    other.stats.getPending());
            }
            if (cmp == 0) {
                cmp = Integer.compare(
                    stats.getAvailable(),
                    other.stats.getAvailable());
            }
            if (cmp == 0) {
                cmp = route.compareTo(other.route);
            }
            return cmp;
        }

        @Override
        public int hashCode() {
            return route.hashCode() ^ stats.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof RouteWithStats) {
                RouteWithStats other = (RouteWithStats) o;
                return route.equals(other.route)
                    && stats.equals(other.stats);
            }
            return false;
        }
    }

    private class PoolRequest extends BasicFuture<NHttpClientConnection> {
        private final long start = TimeSource.INSTANCE.currentTimeMillis();
        private final HttpRoute route;
        private final long connectTimeout;
        private final long connectionRequestTimeout;
        private Future<?> connectionRequest = null;

        // CSOFF: ParameterNumber
        PoolRequest(
            final HttpRoute route,
            final long connectTimeout,
            final long connectionRequestTimeout,
            final FutureCallback<? super NHttpClientConnection> callback)
        {
            super(callback);
            this.route = route;
            this.connectTimeout = connectTimeout;
            this.connectionRequestTimeout = connectionRequestTimeout;
        }
        // CSON: ParameterNumber

        public FutureCallback<InetAddress> dnsCallback() {
            return new DnsCallback();
        }

        @Override
        protected boolean onCancel(final boolean mayInterruptIfRunning) {
            Future<?> connectionRequest;
            synchronized (this) {
                connectionRequest = this.connectionRequest;
                this.connectionRequest = null;
            }
            boolean result = super.onCancel(mayInterruptIfRunning);
            if (connectionRequest != null
                && !connectionRequest.cancel(mayInterruptIfRunning))
            {
                result = false;
            }
            return result;
        }

        private class DnsCallback implements FutureCallback<InetAddress> {
            @Override
            public void failed(final Exception e) {
                pendingPools.decrementAndGet();
                PoolRequest.this.failed(e);
            }

            // Never be called
            @Override
            public void cancelled() {
                pendingPools.decrementAndGet();
                PoolRequest.this.cancelled();
            }

            @Override
            public void completed(final InetAddress address) {
                pendingPools.decrementAndGet();
                RoutePool pool = routes.get(route);
                if (pool == null) {
                    pool = new RoutePool(
                        AsyncNHttpClientConnectionManager.this,
                        route,
                        address);
                    RoutePool oldPool = routes.putIfAbsent(route, pool);
                    if (oldPool == null) {
                        reactor.dnsUpdater().subscribe(
                            pool.targetHost().getHostName(),
                            pool);
                    } else {
                        pool = oldPool;
                    }
                }

                long connectTimeout = PoolRequest.this.connectTimeout;
                if (connectTimeout != 0L) {
                    connectTimeout += start;
                    connectTimeout -= TimeSource.INSTANCE.currentTimeMillis();
                    if (connectTimeout <= 0L) {
                        PoolRequest.this.failed(
                            new DnsTimeoutException(
                                "DNS resolving taken too long"));
                        return;
                    }
                }

                Future<?> connectionRequest = pool.requestConnection(
                    connectTimeout,
                    connectionRequestTimeout,
                    PoolRequest.this);
                synchronized (PoolRequest.this) {
                    if (!done) {
                        PoolRequest.this.connectionRequest = connectionRequest;
                        return;
                    }
                }
                connectionRequest.cancel(true);
            }
        }
    }
}

