package ru.yandex.http.util.nio.client.pool;

import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.util.concurrent.TimeoutException;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.NHttpClientConnection;
import org.apache.http.nio.reactor.IOSession;
import org.apache.http.nio.reactor.SessionRequest;
import org.apache.http.nio.reactor.SessionRequestCallback;

import ru.yandex.concurrent.FutureBase;
import ru.yandex.util.timesource.TimeSource;

public class ConnectionRequest
    extends FutureBase<NHttpClientConnection>
    implements SessionRequestCallback
{
    private final RoutePool pool;
    private final int connectTimeout;
    private final long bornDate;
    private final long connectionRequestTimeout;
    private final long deadline;
    private final FutureCallback<NHttpClientConnection> callback;
    private long connectStart = 0L;
    private SessionRequest sessionRequest = null;

    // CSOFF: ParameterNumber
    public ConnectionRequest(
        final RoutePool pool,
        final long connectTimeout,
        final long bornDate,
        final long connectionRequestTimeout,
        final long deadline,
        final FutureCallback<NHttpClientConnection> callback)
    {
        this.pool = pool;
        if (connectTimeout > Integer.MAX_VALUE) {
            this.connectTimeout = 0;
        } else {
            this.connectTimeout = (int) connectTimeout;
        }
        this.bornDate = bornDate;
        this.connectionRequestTimeout = connectionRequestTimeout;
        this.deadline = deadline;
        this.callback = callback;
    }
    // CSON: ParameterNumber

    public long deadline() {
        return deadline;
    }

    // CSOFF: ReturnCount
    public boolean connect(final InetSocketAddress remoteAddress) {
        long now = TimeSource.INSTANCE.currentTimeMillis();
        synchronized (this) {
            if (done || sessionRequest != null) {
                return false;
            }
            connectStart = now;
            sessionRequest = pool.connManager().connect(
                remoteAddress,
                pool.localAddress(),
                this);
        }
        sessionRequest.setConnectTimeout(connectTimeout);
        return true;
    }
    // CSON: ReturnCount

    // Called by the connection manager to indicate that request deadline has
    // passed. Simple call to `failed(new TimeoutException())' can't be used,
    // because it will decrement leased count, which hasn't been incremented
    public void timedout(final long now) {
        synchronized (this) {
            if (done) {
                return;
            }
            done = true;
        }
        e = new TimeoutException(
            "Dealine " + deadline
            + " passed. Approximate timestamp: " + now
            + ", exact timestamp: " + System.currentTimeMillis()
            + ", connection requested at " + bornDate
            + ", with connection request timeout: "
            + connectionRequestTimeout);
        callback.failed(e);
        synchronized (this) {
            completed = true;
            notifyAll();
        }
    }

    @Override
    protected boolean onCancel(final boolean mayInterruptIfRunning) {
        callback.cancelled();
        boolean result;
        if (sessionRequest != null) {
            // connect(...) has been called on this connection request
            // notify connection pool that this connection request no longer
            // actual, so leased count can be decremented and next pending
            // connection request executed
            pool.connectionRequestFailed();
            sessionRequest.cancel();
            IOSession session = sessionRequest.getSession();
            if (session == null) {
                result = true;
            } else {
                result = false;
                session.close();
            }
        } else {
            result = super.onCancel(mayInterruptIfRunning);
        }
        return result;
    }

    @Override
    protected void onComplete(final NHttpClientConnection conn) {
        PooledNHttpClientConnection connection =
            (PooledNHttpClientConnection) conn;
        TimedConnection timedConn = new TimedConnection(
            connection,
            bornDate,
            connectStart,
            TimeSource.INSTANCE.currentTimeMillis());
        connection.connectionLeased();
        callback.completed(timedConn);
    }

    @Override
    protected void onFailure(final Exception e) {
        if (sessionRequest != null) {
            // connect(...) has been called on this connection request
            // notify connection pool that this connection request no longer
            // actual, so leased count can be decremented and next pending
            // connection request executed
            pool.connectionRequestFailed();
        }
        callback.failed(e);
    }

    public boolean completed(final NHttpClientConnection conn) {
        return completedInternal(conn);
    }

    // SessionRequestCallback implementation
    @Override
    public void completed(final SessionRequest request) {
        PooledNHttpClientConnection conn =
            pool.connManager().createConnection(pool, request.getSession());
        if (!completedInternal(conn)) {
            conn.closeQuietly();
        }
    }

    @Override
    public void failed(final SessionRequest request) {
        failed(request.getException());
    }

    @Override
    public void timeout(final SessionRequest request) {
        failed(new ConnectException("Failed to connect to host"));
    }

    @Override
    public void cancelled(final SessionRequest request) {
        // Yeah, I know
    }
}

