package ru.yandex.http.util.nio.client.pool;

import java.net.InetAddress;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.function.Supplier;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.function.GenericAutoCloseableChain;

public class MultiDnsResolver implements AsyncDnsResolver {
    private AsyncDnsResolver[] resolvers;

    public MultiDnsResolver(
        final int count,
        final Supplier<? extends AsyncDnsResolver> resolverFactory)
    {
        resolvers = new AsyncDnsResolver[count];
        for (int i = 0; i < count; ++i) {
            resolvers[i] = resolverFactory.get();
        }
    }

    @Override
    public void start() {
        for (AsyncDnsResolver resolver: resolvers) {
            resolver.start();
        }
    }

    @Override
    public void close() {
        try (GenericAutoCloseableChain<RuntimeException> chain =
                new GenericAutoCloseableChain<>())
        {
            for (AsyncDnsResolver resolver: resolvers) {
                chain.add(resolver);
            }
        }
    }

    @Override
    public Future<InetAddress> resolve(
        final String hostname,
        final FutureCallback<InetAddress> callback)
    {
        int id = hostname.hashCode() % resolvers.length;
        if (id < 0) {
            id += resolvers.length;
        }
        return resolvers[id].resolve(hostname, callback);
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        for (int i = 0; i < resolvers.length; ++i) {
            Map<String, Object> resolverStatus = resolvers[i].status(verbose);
            if (!resolverStatus.isEmpty()) {
                status.put(Integer.toString(i), resolverStatus);
            }
        }
        return status;
    }
}

