package ru.yandex.http.util.nio.client.pool;

import java.io.IOException;

import org.apache.http.HttpHost;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.impl.ConnSupport;
import org.apache.http.impl.nio.DefaultNHttpClientConnection;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.nio.reactor.IOSession;

public class PooledNHttpClientConnection extends DefaultNHttpClientConnection {
    private final RoutePool pool;
    private final int defaultSocketTimeout;
    private volatile boolean isRouteComplete;
    private volatile long validUntil;
    private volatile long idleSince;

    public PooledNHttpClientConnection(
        final IOSession session,
        final ConnectionConfig config,
        final RoutePool pool)
    {
        super(
            session,
            config.getBufferSize(),
            config.getFragmentSizeHint(),
            null,
            ConnSupport.createDecoder(config),
            ConnSupport.createEncoder(config),
            config.getMessageConstraints(),
            null,
            null,
            null,
            null);
        this.pool = pool;
        defaultSocketTimeout = getSocketTimeout();
    }

    public boolean isRouteComplete() {
        return isRouteComplete;
    }

    public void markRouteComplete() {
        isRouteComplete = true;
    }

    public void updateIdle(final long idleSince, final long keepalive) {
        this.idleSince = idleSince;
        if (keepalive <= 0L) {
            validUntil = Long.MAX_VALUE;
        } else {
            validUntil = idleSince + keepalive;
        }
    }

    public long validUntil() {
        return validUntil;
    }

    public long idleSince() {
        return idleSince;
    }

    public void releaseConnection(final long keepalive) {
        setSocketTimeout(0);
        pool.releaseConnection(this, keepalive);
    }

    public void connectionLeased() {
        setSocketTimeout(defaultSocketTimeout);
    }

    public void closeQuietly() {
        // CSOFF: EmptyBlock
        try {
            close();
        } catch (IOException e) {
        }
        // CSON: EmptyBlock
    }

    public void startRoute(final HttpHost targetHost) throws IOException {
        SchemeIOSessionStrategy ioSessionStrategy = pool.ioSessionStrategy();
        if (ioSessionStrategy.isLayeringRequired()) {
            bind(ioSessionStrategy.upgrade(targetHost, session));
        }
    }
}

