package ru.yandex.http.util.nio.client.pool;

import java.io.Closeable;
import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;

import org.apache.http.HttpHost;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.nio.NHttpClientConnection;
import org.apache.http.nio.conn.SchemeIOSessionStrategy;
import org.apache.http.pool.PoolStats;

import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.concurrent.ConcurrentStackStorage;
import ru.yandex.concurrent.FailedFuture;
import ru.yandex.concurrent.ObjectsStorage;
import ru.yandex.util.timesource.TimeSource;

public class RoutePool implements Closeable, Consumer<InetAddress> {
    private final ObjectsStorage<PooledNHttpClientConnection> available =
        new ConcurrentStackStorage<>();
    private final Queue<ConnectionRequest> pending =
        new ConcurrentLinkedQueue<>();
    private final AtomicBoolean closed = new AtomicBoolean();
    private final AtomicInteger leased = new AtomicInteger();
    private final AsyncNHttpClientConnectionManager connManager;
    private final HttpHost targetHost;
    private final int port;
    private final InetSocketAddress localAddress;
    private final SchemeIOSessionStrategy ioSessionStrategy;
    private volatile InetSocketAddress remoteAddress;
    private int addressResolutionFailures = 0;

    // CSOFF: ParameterNumber
    public RoutePool(
        final AsyncNHttpClientConnectionManager connManager,
        final HttpRoute route,
        final InetAddress remoteAddress)
    {
        this.connManager = connManager;
        targetHost = AsyncNHttpClientConnectionManager.targetHost(route);
        port = targetHost.getPort();
        InetAddress localAddress = route.getLocalAddress();
        if (localAddress == null) {
            this.localAddress = null;
        } else {
            this.localAddress = new InetSocketAddress(localAddress, 0);
        }
        ioSessionStrategy = connManager.selectIOSessionStrategy(targetHost);
        this.remoteAddress = new InetSocketAddress(remoteAddress, port);
    }
    // CSON: ParameterNumber

    public AsyncNHttpClientConnectionManager connManager() {
        return connManager;
    }

    public HttpHost targetHost() {
        return targetHost;
    }

    public InetSocketAddress localAddress() {
        return localAddress;
    }

    public SchemeIOSessionStrategy ioSessionStrategy() {
        return ioSessionStrategy;
    }

    // CSOFF: FinalParameters
    private void put(
        PooledNHttpClientConnection conn,
        ConnectionRequest request,
        final long now)
    {
        while (true) {
            if (conn == null) {
                if (request == null) {
                    break;
                } else {
                    conn = available.get();
                    if (conn == null) {
                        pending.add(request);
                        request = null;
                        conn = available.get();
                    }
                }
            } else if (conn.validUntil() < now) {
                conn.closeQuietly();
                conn = available.get();
                if (request == null) {
                    request = pending.poll();
                }
            } else if (request == null) {
                available.put(conn);
                conn = null;
                request = pending.poll();
            } else if (request.deadline() < now) {
                request.timedout(now);
            } else if (request.completed(conn)) {
                leased.getAndIncrement();
                break;
            } else {
                request = pending.poll();
            }
        }
    }
    // CSON: FinalParameters

    // CSOFF: ReturnCount
    public Future<NHttpClientConnection> requestConnection(
        final long connectTimeout,
        final long connectionRequestTimeout,
        final FutureCallback<NHttpClientConnection> callback)
    {
        InetSocketAddress remoteAddress = this.remoteAddress;
        if (remoteAddress == null) {
            Exception e = new UnknownHostException();
            callback.failed(e);
            return new FailedFuture<>(e);
        }
        long now = TimeSource.INSTANCE.currentTimeMillis();
        PooledNHttpClientConnection conn = available.get();
        while (conn != null) {
            if (conn.isOpen() && conn.validUntil() > now) {
                TimedConnection timedConn =
                    new TimedConnection(conn, now, now, now);
                leased.getAndIncrement();
                conn.connectionLeased();
                callback.completed(timedConn);
                return new CompletedFuture<NHttpClientConnection>(timedConn);
            } else {
                conn.closeQuietly();
                conn = available.get();
            }
        }
        long deadline;
        if (leased.getAndIncrement() < connManager.getDefaultMaxPerRoute()) {
            deadline = 0L;
        } else {
            leased.decrementAndGet();
            if (connectionRequestTimeout == 0L) {
                deadline = Long.MAX_VALUE;
            } else {
                deadline = now + connectionRequestTimeout;
            }
        }
        ConnectionRequest request =
            new ConnectionRequest(
                this,
                connectTimeout,
                now,
                connectionRequestTimeout,
                deadline,
                callback);
        if (deadline == 0L) {
            request.connect(remoteAddress);
        } else {
            put(null, request, now);
        }
        return request;
    }
    // CSON: ReturnCount

    private void connectNextPendingRequest(final long now) {
        ConnectionRequest request = pending.poll();
        if (request != null) {
            InetSocketAddress remoteAddress = this.remoteAddress;
            if (remoteAddress == null) {
                List<ConnectionRequest> requests = new ArrayList<>();
                do {
                    requests.add(request);
                    request = pending.poll();
                } while (request != null);
                UnknownHostException e = new UnknownHostException();
                requests.forEach(r -> r.failed(e));
            } else {
                do {
                    if (request.deadline() > now) {
                        if (request.connect(remoteAddress)) {
                            leased.getAndIncrement();
                            break;
                        }
                    } else {
                        request.timedout(now);
                    }
                    request = pending.poll();
                } while (request != null);
            }
        }
    }

    public void releaseConnection(
        final PooledNHttpClientConnection conn,
        final long keepalive)
    {
        leased.decrementAndGet();
        long now = TimeSource.INSTANCE.currentTimeMillis();
        if (conn.isOpen() && conn.isRouteComplete()) {
            if (!processNextPendingRequest(conn, now)) {
                conn.updateIdle(now, keepalive);
                put(conn, null, now);
            }
        } else {
            conn.closeQuietly();
            connectNextPendingRequest(now);
        }
    }

    public void connectionRequestFailed() {
        leased.decrementAndGet();
        connectNextPendingRequest(TimeSource.INSTANCE.currentTimeMillis());
    }

    private boolean processNextPendingRequest(
        final PooledNHttpClientConnection conn,
        final long now)
    {
        ConnectionRequest request = pending.poll();
        while (request != null) {
            if (request.deadline() > now) {
                if (request.completed(conn)) {
                    // We are not checking for < getDefaultMaxPerRoute because
                    // this can lead to non-fair requests processing
                    leased.getAndIncrement();
                    return true;
                }
            } else {
                request.timedout(now);
            }
            request = pending.poll();
        }
        return false;
    }

    private void processPendingRequests(final long now) {
        PooledNHttpClientConnection conn = available.get();
        while (conn != null && processNextPendingRequest(conn, now)) {
            conn = available.get();
        }
        if (conn != null) {
            put(conn, null, now);
        }
    }

    private void processPendingRequests(
        final List<PooledNHttpClientConnection> connections,
        final long now)
    {
        int pos = 0;
        int size = connections.size();
        while (pos < size) {
            PooledNHttpClientConnection conn = connections.get(pos);
            if (conn.validUntil() > now) {
                if (!processNextPendingRequest(conn, now)) {
                    break;
                }
            } else {
                conn.closeQuietly();
            }
            ++pos;
        }
        if (pos < size) {
            available.put(connections.get(pos++));
            while (pos < size) {
                available.put(connections.get(pos++));
            }
            processPendingRequests(now);
        }
    }

    public void closeIdleConnections(final long idleSince) {
        List<PooledNHttpClientConnection> connections = new ArrayList<>();
        PooledNHttpClientConnection conn = available.get();
        while (conn != null) {
            if (conn.idleSince() > idleSince) {
                connections.add(conn);
            } else {
                conn.closeQuietly();
            }
            conn = available.get();
        }
        if (!connections.isEmpty()) {
            processPendingRequests(
                connections,
                TimeSource.INSTANCE.currentTimeMillis());
        }
    }

    public void closeExpiredConnections(final long now) {
        List<PooledNHttpClientConnection> connections = new ArrayList<>();
        PooledNHttpClientConnection conn = available.get();
        while (conn != null) {
            if (conn.validUntil() > now) {
                connections.add(conn);
            } else {
                conn.closeQuietly();
            }
            conn = available.get();
        }
        if (!connections.isEmpty()) {
            processPendingRequests(connections, now);
        }
    }

    public PoolStats stats() {
        return new PoolStats(
            leased.get(),
            pending.size(),
            available.size(),
            connManager.getDefaultMaxPerRoute());
    }

    @Override
    public void close() throws IOException {
        if (closed.compareAndSet(false, true)) {
            ConnectionRequest request = pending.poll();
            while (request != null) {
                request.cancel(true);
                request = pending.poll();
            }
            PooledNHttpClientConnection conn = available.get();
            while (conn != null) {
                conn.close();
                conn = available.get();
            }
        }
    }

    @Override
    public void accept(final InetAddress remoteAddress) {
        if (remoteAddress == null) {
            ++addressResolutionFailures;
            if (addressResolutionFailures > 2) {
                this.remoteAddress = null;
            }
        } else {
            addressResolutionFailures = 0;
            if (this.remoteAddress == null
                || !remoteAddress.equals(this.remoteAddress.getAddress()))
            {
                this.remoteAddress =
                    new InetSocketAddress(remoteAddress, port);
            }
        }
    }
}

