package ru.yandex.http.util.nio.client.pool;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.conn.DnsResolver;

import ru.yandex.concurrent.CompletedFuture;
import ru.yandex.http.util.BasicFuture;
import ru.yandex.util.timesource.TimeSource;

public class SingleThreadDnsResolver implements AsyncDnsResolver, Runnable {
    public static final String CACHE_SIZE = "cache_size";
    public static final String RESOLVING = "resolving";
    public static final String PENDING_COUNT = "pending_count";
    public static final String PENDING_REQUESTS = "pending_requests";

    private final Lock lock = new ReentrantLock();
    private final Condition cond = lock.newCondition();
    private final Map<String, Node> cache = new ConcurrentHashMap<>();
    private final DnsResolver dnsResolver;
    private final long ttl;
    private final long updateInterval;
    private final Thread thread;
    private volatile String currentHostname = null;
    private volatile boolean hasNewRequests = false;
    private boolean stopped = false;
    private List<Request> requests = new ArrayList<>();
    private List<Request> inactiveRequests = new ArrayList<>();

    // CSOFF: ParameterNumber
    public SingleThreadDnsResolver(
        final DnsResolver dnsResolver,
        final long ttl,
        final long updateInterval,
        final ThreadFactory threadFactory)
    {
        this.dnsResolver = dnsResolver;
        this.ttl = ttl;
        this.updateInterval = updateInterval;
        thread = threadFactory.newThread(this);
    }
    // CSON: ParameterNumber

    @Override
    public void start() {
        thread.start();
    }

    @Override
    public void close() {
        lock.lock();
        try {
            stopped = true;
            cond.signal();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public Future<InetAddress> resolve(
        final String hostname,
        final FutureCallback<InetAddress> callback)
    {
        Node node = cache.get(hostname);
        if (node == null) {
            BasicFuture<InetAddress> future = new BasicFuture<>(callback);
            lock.lock();
            try {
                requests.add(new Request(hostname, future));
                hasNewRequests = true;
                cond.signal();
            } finally {
                lock.unlock();
            }
            return future;
        } else {
            InetAddress address = node.address();
            callback.completed(address);
            return new CompletedFuture<>(address);
        }
    }

    private InetAddress resolve(final String hostname)
        throws UnknownHostException
    {
        return dnsResolver.resolve(hostname)[0];
    }

    @Override
    @SuppressWarnings("MixedMutabilityReturnType")
    public Map<String, Object> status(final boolean verbose) {
        List<Request> requests;
        lock.lock();
        try {
            requests = new ArrayList<>(this.requests);
        } finally {
            lock.unlock();
        }
        String currentHostname = this.currentHostname;
        if (currentHostname == null && requests.isEmpty()) {
            return Collections.emptyMap();
        }
        Map<String, Object> status = new LinkedHashMap<>();
        status.put(CACHE_SIZE, cache.size());
        if (currentHostname != null) {
            status.put(RESOLVING, currentHostname);
        }
        status.put(PENDING_COUNT, requests.size());
        if (verbose && !requests.isEmpty()) {
            Map<String, Integer> perHost = new TreeMap<>();
            for (Request request: requests) {
                String hostname = request.hostname();
                Integer value = perHost.get(hostname);
                if (value == null) {
                    value = 1;
                } else {
                    ++value;
                }
                perHost.put(hostname, value);
            }
            status.put(PENDING_REQUESTS, perHost);
        }
        return status;
    }

    @Override
    public void run() {
        Map<String, UnknownHostException> failures = new HashMap<>();
        long lastUpdated = TimeSource.INSTANCE.currentTimeMillis();
        while (!stopped) {
            List<Request> requests;
            lock.lock();
            try {
                if (this.requests.isEmpty()) {
                    try {
                        cond.await(updateInterval, TimeUnit.MILLISECONDS);
                    } catch (InterruptedException e) {
                        break;
                    }
                }
                requests = this.requests;
                this.requests = inactiveRequests;
                inactiveRequests = requests;
                hasNewRequests = false;
            } finally {
                lock.unlock();
            }
            long now = TimeSource.INSTANCE.currentTimeMillis();
            for (Request request: requests) {
                String hostname = request.hostname();
                Node node = cache.get(hostname);
                if (node == null) {
                    UnknownHostException exc = failures.get(hostname);
                    if (exc == null) {
                        currentHostname = hostname;
                        try {
                            InetAddress address = resolve(hostname);
                            request.callback().completed(address);
                            node = new Node();
                            node.addressResolved(address, now);
                            cache.put(hostname, node);
                        } catch (UnknownHostException e) {
                            request.callback().failed(e);
                            failures.put(hostname, e);
                        }
                        currentHostname = null;
                    } else {
                        request.callback().failed(exc);
                    }
                } else {
                    request.callback().completed(node.address());
                }
            }
            requests.clear();
            failures.clear();
            long minLastUpdated = now - updateInterval;
            if (lastUpdated < minLastUpdated) {
                long minLastUsed = now - ttl;
                Iterator<Map.Entry<String, Node>> iter =
                    cache.entrySet().iterator();
                while (iter.hasNext() && !hasNewRequests) {
                    Map.Entry<String, Node> entry = iter.next();
                    Node node = entry.getValue();
                    if (node.lastUsed() < minLastUsed) {
                        iter.remove();
                    } else if (node.lastUpdated() < minLastUpdated) {
                        try {
                            node.addressUpdated(resolve(entry.getKey()), now);
                        } catch (UnknownHostException e) {
                            iter.remove();
                        }
                    }
                }
                if (!hasNewRequests) {
                    lastUpdated = now;
                }
            }
        }
    }

    private static class Node {
        private volatile InetAddress address;
        private volatile long lastUsed;
        private long lastUpdated;

        public void addressResolved(
            final InetAddress address,
            final long lastUsed)
        {
            this.address = address;
            this.lastUsed = lastUsed;
            lastUpdated = lastUsed;
        }

        public void addressUpdated(
            final InetAddress address,
            final long lastUpdated)
        {
            this.address = address;
            this.lastUpdated = lastUpdated;
        }

        public InetAddress address() {
            lastUsed = TimeSource.INSTANCE.currentTimeMillis();
            return address;
        }

        public long lastUsed() {
            return lastUsed;
        }

        public long lastUpdated() {
            return lastUpdated;
        }
    }

    private static class Request {
        private final String hostname;
        private final FutureCallback<? super InetAddress> callback;

        Request(
            final String hostname,
            final FutureCallback<? super InetAddress> callback)
        {
            this.hostname = hostname;
            this.callback = callback;
        }

        public String hostname() {
            return hostname;
        }

        public FutureCallback<? super InetAddress> callback() {
            return callback;
        }
    }
}

