package ru.yandex.http.util.nio.client.pool;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ThreadFactory;
import java.util.function.Consumer;

import org.apache.http.conn.DnsResolver;

public class SingleThreadDnsUpdater implements DnsUpdater, Runnable {
    private final ConcurrentMap<String, List<Consumer<InetAddress>>>
        subscribers = new ConcurrentHashMap<>();
    private final DnsResolver dnsResolver;
    private final long updateInterval;
    private final Thread thread;
    private boolean stopped = false;

    public SingleThreadDnsUpdater(
        final DnsResolver dnsResolver,
        final long updateInterval,
        final ThreadFactory threadFactory)
    {
        this.dnsResolver = dnsResolver;
        this.updateInterval = updateInterval;
        thread = threadFactory.newThread(this);
    }

    @Override
    public void start() {
        thread.start();
    }

    @Override
    public void subscribe(
        final String hostname,
        final Consumer<InetAddress> subscriber)
    {
        List<Consumer<InetAddress>> subscribers =
            this.subscribers.get(hostname);
        if (subscribers == null) {
            subscribers = new ArrayList<>();
            List<Consumer<InetAddress>> oldSubscribers =
                this.subscribers.putIfAbsent(hostname, subscribers);
            if (oldSubscribers != null) {
                subscribers = oldSubscribers;
            }
        }
        synchronized (subscribers) {
            subscribers.add(subscriber);
        }
    }

    private void update() {
        for (Map.Entry<String, List<Consumer<InetAddress>>> entry
            : subscribers.entrySet())
        {
            InetAddress address;
            try {
                address = dnsResolver.resolve(entry.getKey())[0];
            } catch (UnknownHostException e) {
                address = null;
            }
            List<Consumer<InetAddress>> subscribers = entry.getValue();
            synchronized (subscribers) {
                for (Consumer<InetAddress> subscriber: subscribers) {
                    subscriber.accept(address);
                }
            }
        }
    }

    @Override
    public void run() {
        while (!stopped) {
            synchronized (this) {
                if (!stopped) {
                    try {
                        wait(updateInterval);
                    } catch (InterruptedException e) {
                        break;
                    }
                }
            }
            update();
        }
    }

    @Override
    public synchronized void close() {
        stopped = true;
        notify();
    }
}

