package ru.yandex.http.util.client;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HostnameVerifier;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.protocol.RequestAcceptEncoding;
import org.apache.http.client.protocol.ResponseContentEncoding;
import org.apache.http.config.SocketConfig;
import org.apache.http.conn.DnsResolver;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.conn.ssl.DefaultHostnameVerifier;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.NoConnectionReuseStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.impl.conn.SystemDefaultDnsResolver;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpProcessorBuilder;
import org.apache.http.protocol.RequestContent;

import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.http.config.ImmutableClientHttpsConfig;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.http.config.ImmutableHttpTargetConfig;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.ServerErrorStatusPredicate;
import ru.yandex.parser.config.ConfigException;

public final class ClientBuilder {
    private static final HostnameVerifier DEFAULT_HOSTNAME_VERIFIER =
        new DefaultHostnameVerifier();

    private ClientBuilder() {
    }

    public static SocketConfig createSocketConfig(
        final ImmutableHttpTargetConfig config)
    {
        return SocketConfig
            .custom()
            .setSoTimeout(config.timeout())
            .setSoReuseAddress(true)
            .setSoKeepAlive(true)
            .setTcpNoDelay(true)
            .setSoLinger(0)
            .build();
    }

    public static TimingsPoolingHttpClientConnectionManager createConnManager(
        final ImmutableHttpTargetConfig backendConfig,
        final ImmutableDnsConfig dnsConfig)
    {
        TimingsPoolingHttpClientConnectionManager connManager =
            new TimingsPoolingHttpClientConnectionManager(
                createSSLConnectionSocketFactory(backendConfig.httpsConfig()),
                createDnsResolver(dnsConfig));
        connManager.setDefaultMaxPerRoute(backendConfig.connections());
        connManager.setMaxTotal(Integer.MAX_VALUE);
        connManager.setDefaultConnectionConfig(
            backendConfig.toConnectionConfig());
        connManager.setDefaultSocketConfig(createSocketConfig(backendConfig));
        return connManager;
    }

    public static String[] enabledProtocols(
        final ImmutableClientHttpsConfig config)
    {
        List<String> enabledProtocols = config.enabledProtocols();
        if (enabledProtocols.isEmpty()) {
            return null;
        } else {
            return enabledProtocols.toArray(
                new String[enabledProtocols.size()]);
        }
    }

    public static HostnameVerifier createHostnameVerifier(
        final ImmutableClientHttpsConfig config)
    {
        if (config.verifyHostname()) {
            return DEFAULT_HOSTNAME_VERIFIER;
        } else {
            return NoopHostnameVerifier.INSTANCE;
        }
    }

    public static SSLConnectionSocketFactory createSSLConnectionSocketFactory(
        final ImmutableClientHttpsConfig config)
    {
        return new SSLConnectionSocketFactory(
            config.getSSLContext(),
            enabledProtocols(config),
            null,
            createHostnameVerifier(config));
    }

    // TODO: Remove after HttpComponents upgrade to 5.x branch
    public static RequestConfig createRequestConfig(
        final ImmutableHttpTargetConfig config)
    {
        return createRequestConfig(config, true);
    }

    @SuppressWarnings("deprecation")
    public static RequestConfig createRequestConfig(
        final ImmutableHttpTargetConfig config,
        final boolean compressionSupported)
    {
        return RequestConfig
            .custom()
            .setProxy(config.proxy())
            .setConnectTimeout(config.connectTimeout())
            .setSocketTimeout(config.timeout())
            .setConnectionRequestTimeout(config.poolTimeout())
            .setContentCompressionEnabled(
                compressionSupported && config.contentCompression())
            .setRedirectsEnabled(config.redirects())
            .setRelativeRedirectsAllowed(config.relativeRedirects())
            .setCircularRedirectsAllowed(config.circularRedirects())
            .setMaxRedirects(config.maxRedirects())
            .setStaleConnectionCheckEnabled(false)
            .setAuthenticationEnabled(false)
            .build();
    }

    public static HttpProcessorBuilder prepareHttpProcessorBuilder(
        final ImmutableHttpTargetConfig config)
    {
        return prepareHttpProcessorBuilder(config, true);
    }

    public static HttpProcessorBuilder prepareHttpProcessorBuilder(
        final ImmutableHttpTargetConfig config,
        final boolean compressionSupported)
    {
        HttpProcessorBuilder processorBuilder = HttpProcessorBuilder.create();
        processorBuilder.add(RequestTargetHost.INSTANCE);
        processorBuilder.add(new RequestContent());

        List<Header> headers = new ArrayList<>(2);
        if (!config.briefHeaders()) {
            headers.add(
                HeaderUtils.createHeader(
                    HttpHeaders.ACCEPT_CHARSET,
                    config.responseCharset().name()));
        }
        if (!config.keepAlive()) {
            headers.add(
                HeaderUtils.createHeader(
                    HTTP.CONN_DIRECTIVE,
                    HTTP.CONN_CLOSE));
        }
        if (!headers.isEmpty()) {
            processorBuilder.add(new RequestDefaultHeaders(headers));
        }

        if (compressionSupported && config.contentCompression()) {
            processorBuilder.add(new RequestAcceptEncoding());
            processorBuilder.add(new ResponseContentEncoding());
        }

        return processorBuilder;
    }

    public static DnsResolver createDnsResolver(
        final ImmutableDnsConfig config)
    {
        Map<String, String> mapping = config.dnsHostsMapping();
        if (mapping.isEmpty()) {
            return SystemDefaultDnsResolver.INSTANCE;
        } else {
            return new MapDnsResolver(
                SystemDefaultDnsResolver.INSTANCE,
                mapping);
        }
    }

    @Deprecated
    public static CloseableHttpClient createClient(
        final ImmutableHttpTargetConfig backendConfig)
    {
        try {
            return createClient(
                backendConfig,
                new DnsConfigBuilder().build());
        } catch (ConfigException e) {
            throw new RuntimeException("Impossible", e);
        }
    }

    public static CloseableHttpClient createClient(
        final ImmutableHttpTargetConfig backendConfig,
        final ImmutableDnsConfig dnsConfig)
    {
        return createClient(
            backendConfig,
            createConnManager(backendConfig, dnsConfig));
    }

    public static CloseableHttpClient createClient(
        final ImmutableHttpTargetConfig config,
        final HttpClientConnectionManager connManager)
    {
        HttpClientBuilder builder = HttpClients.custom()
            .disableConnectionState()
            .setConnectionManager(connManager)
            .setDefaultRequestConfig(createRequestConfig(config))
            .setHttpProcessor(prepareHttpProcessorBuilder(config).build())
            .setRoutePlanner(new PlainHttpRoutePlanner(config.proxy()));

        if (connManager instanceof TimingsPoolingHttpClientConnectionManager) {
            builder.setRequestExecutor(new TimingsHttpRequestExecutor());
        } else if (connManager == null) {
            builder.setDefaultConnectionConfig(config.toConnectionConfig());
            builder.setDefaultSocketConfig(createSocketConfig(config));
        }
        if (connManager instanceof PoolingHttpClientConnectionManager) {
            ((PoolingHttpClientConnectionManager) connManager)
                .setDefaultConnectionConfig(config.toConnectionConfig());
            ((PoolingHttpClientConnectionManager) connManager)
                .setDefaultSocketConfig(createSocketConfig(config));
            ((PoolingHttpClientConnectionManager) connManager)
                .setValidateAfterInactivity(0);
        }
        if (config.ioRetries().count() == 0) {
            builder.disableAutomaticRetries();
        } else {
            builder.setRetryHandler(
                new DelayHttpRequestRetryHandler(config.ioRetries()));
        }
        if (config.httpRetries().count() != 0) {
            builder.setServiceUnavailableRetryStrategy(
                new BasicServiceUnavailableRetryStrategy(
                    config.httpRetries(),
                    ServerErrorStatusPredicate.INSTANCE));
        }
        if (config.keepAlive()) {
            builder.setConnectionReuseStrategy(
                DefaultConnectionReuseStrategy.INSTANCE);
        } else {
            builder.setConnectionReuseStrategy(
                NoConnectionReuseStrategy.INSTANCE);
        }
        if (config.redirects()) {
            if (config.redirectPosts()) {
                builder.setRedirectStrategy(new PostsRedirectStrategy());
            }
        } else {
            builder.disableRedirectHandling();
        }
        return builder.build();
    }
}

