package ru.yandex.http.util.client.measurable;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Map;
import java.util.TreeMap;

import org.apache.http.protocol.HttpContext;

import ru.yandex.util.timesource.TimeSource;

// TODO introduce tests
public class Measurer {
    public static final String MEASURER =
        "ru.yandex.http.util.client.measurable.measurer";

    private static final String ARROW = " -> ";

    public enum Scope {
        EXECUTE,
        POOL_GET,
        CONNECT,
        SEND,
        RECEIVE,
    }

    public enum Bound {
        BEFORE,
        AFTER,
    }

    public enum Event {
        BEFORE_EXECUTE(Scope.EXECUTE, Bound.BEFORE),
        AFTER_EXECUTE(Scope.EXECUTE, Bound.AFTER),
        BEFORE_POOL_GET(Scope.POOL_GET, Bound.BEFORE),
        AFTER_POOL_GET(Scope.POOL_GET, Bound.AFTER),
        BEFORE_CONNECT(Scope.CONNECT, Bound.BEFORE),
        AFTER_CONNECT(Scope.CONNECT, Bound.AFTER),
        BEFORE_SEND(Scope.SEND, Bound.BEFORE),
        AFTER_SEND(Scope.SEND, Bound.AFTER),
        BEFORE_RECEIVE(Scope.RECEIVE, Bound.BEFORE),
        AFTER_RECEIVE(Scope.RECEIVE, Bound.AFTER);

        private final Scope scope;
        private final Bound bound;

        Event(final Scope scope, final Bound boundary) {
            this.scope = scope;
            this.bound = boundary;
        }
    }

    private static class Occasion {
        private final Event event;
        private final long time;

        Occasion(final Event e, final long time) {
            this.event = e;
            this.time = time;
        }
    }

    private final Deque<Occasion> data = new ArrayDeque<>();
    private final Map<Scope, Deque<Occasion>> dataByScope = new TreeMap<>();

    public Measurer() {
        snapshot(Event.BEFORE_EXECUTE);
    }

    public void snapshot(final Event event) {
        Long time = TimeSource.INSTANCE.currentTimeMillis();
        data.add(new Occasion(event, time));
        Deque<Occasion> ls = dataByScope.get(event.scope);
        if (ls == null) {
            ls = new ArrayDeque<>();
            dataByScope.put(event.scope, ls);
        }
        ls.add(new Occasion(event, time));
    }

    // TODO: do this automatically
    public void fin() {
        snapshot(Event.AFTER_EXECUTE);
    }

    public String scopePrint() {
        StringBuilder sb = new StringBuilder();
        for (Scope sc: Scope.values()) {
            Deque<Occasion> ocs = dataByScope.get(sc);
            if (ocs == null || ocs.isEmpty()) {
                continue;
            }
            Deque<Long> stack = new ArrayDeque<>();
            for (Occasion entry: ocs) {
                Long ts = entry.time;
                Event ev = entry.event;
                Bound b = ev.bound;
                if (b == Bound.BEFORE) {
                    stack.addLast(ts);
                } else {
                    assert !stack.isEmpty();
                    Long prev = stack.getLast();
                    Long diff = ts - prev;
                    sb.append(ev.scope.toString()
                            + ' '
                            + diff
                            + ' ');
                }
            }
            assert stack.isEmpty();
        }
        sb.append("ms");
        return sb.toString();
    }

    public String fullPrint() {
        StringBuilder sb = new StringBuilder();
        for (Occasion entry: data) {
            sb.append(entry.time);
            sb.append(' ');
            sb.append(entry.event);
            sb.append('\n');
        }
        return new String(sb);
    }

    public String diffPrint() {
        StringBuilder sb = new StringBuilder();
        long prev = 0;
        for (Occasion entry: data) {
            if (prev > 0) {
                sb.append(ARROW);
                sb.append(entry.time - prev);
                sb.append(ARROW);
            }
            sb.append(entry.event);
            prev = entry.time;
        }
        return new String(sb);
    }

    @Override
    public String toString() {
        return scopePrint() + " -- " + diffPrint();
    }

    public static Measurer getMeasurer(final HttpContext context) {
        Object o = context.getAttribute(Measurer.MEASURER);
        Measurer m = null;
        try {
            m = (Measurer) o;
        } catch (ClassCastException e) {
            m = null;
        }
        return m;
    }
}

