package ru.yandex.http.util.request;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.message.BasicHeader;

import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;

public abstract class RequestHandlerMapper<T> {
    public static final String ASTERISK = PatternMap.ASTERISK;
    public static final String GET = "GET";
    public static final String HEAD = "HEAD";
    public static final String POST = "POST";
    public static final String PUT = "PUT";
    public static final String OPTIONS = "OPTIONS";
    public static final String[] DEFAULT_METHODS =
        new String[] {GET, HEAD, POST, PUT};

    private final Map<String, PatternMap<RequestInfo, T>> matchers =
        new LinkedHashMap<>();
    private T globalOptionsHandler = null;

    protected RequestHandlerMapper() {
        matchers.put(ASTERISK, new PatternMap<>());
        matchers.put(GET, new PatternMap<>());
        matchers.put(HEAD, new PatternMap<>());
        matchers.put(POST, new PatternMap<>());
        matchers.put(PUT, new PatternMap<>());
        matchers.put(OPTIONS, new PatternMap<>());
    }

    public void register(final Pattern<RequestInfo> pattern, final T handler) {
        register(pattern, handler, DEFAULT_METHODS);
    }

    public void register(
        final Pattern<RequestInfo> pattern,
        final T handler,
        final String... methods)
    {
        for (String method: methods) {
            register(pattern, handler, method);
        }
    }

    public T register(
        final Pattern<RequestInfo> pattern,
        final T handler,
        final String method)
    {
        T result;
        if (method.equals(OPTIONS) && pattern.isAsterisk()) {
            result = globalOptionsHandler;
            globalOptionsHandler = handler;
        } else {
            PatternMap<RequestInfo, T> matcher = matchers.get(method);
            if (matcher == null) {
                matcher = new PatternMap<>();
                matchers.put(method, matcher);
            }
            result = matcher.put(pattern, handler);
            if (method.equals(ASTERISK)) {
                if (result == null) {
                    Iterator<PatternMap<RequestInfo, T>> iter =
                        matchers.values().iterator();
                    iter.next();
                    while (iter.hasNext()) {
                        matcher = iter.next();
                        T subresult = matcher.remove(pattern);
                        if (subresult != null) {
                            result = subresult;
                        }
                    }
                }
            } else {
                T subresult = matchers.get(ASTERISK).remove(pattern);
                if (subresult != null) {
                    result = subresult;
                }
            }
        }
        return result;
    }

    public void unregister(final Pattern<RequestInfo> pattern) {
        unregister(pattern, ASTERISK);
    }

    public void unregister(
        final Pattern<RequestInfo> pattern,
        final String... methods)
    {
        for (String method: methods) {
            unregister(pattern, method);
        }
    }

    public T unregister(
        final Pattern<RequestInfo> pattern,
        final String method)
    {
        T result;
        if (method.equals(OPTIONS) && pattern.isAsterisk()) {
            result = globalOptionsHandler;
            globalOptionsHandler = null;
        } else if (method.equals(ASTERISK)) {
            result = null;
            for (PatternMap<RequestInfo, T> matcher: matchers.values()) {
                T subresult = matcher.remove(pattern);
                if (subresult != null) {
                    result = subresult;
                }
            }
        } else {
            PatternMap<RequestInfo, T> matcher = matchers.get(method);
            if (matcher == null) {
                result = null;
            } else {
                result = matcher.remove(pattern);
            }
        }
        return result;
    }

    public T lookup(final HttpRequest request) {
        return lookup(new RequestInfo(request));
    }

    public T lookup(final RequestInfo request) {
        T result;
        if (request.method().equals(OPTIONS)) {
            if (request.paths().get(0).equals(ASTERISK)) {
                if (globalOptionsHandler == null) {
                    result = globalOptionsHandler();
                } else {
                    result = globalOptionsHandler;
                }
            } else {
                result = matchers.get(OPTIONS).get(request);
                if (result == null) {
                    result = optionsHandler(request);
                }
            }
        } else {
            PatternMap<RequestInfo, T> matcher =
                matchers.get(request.method());
            if (matcher == null) {
                result = null;
            } else {
                result = matcher.get(request);
            }
            if (result == null) {
                result = matchers.get(ASTERISK).get(request);
            }
            if (result == null) {
                result = notFoundHandler(request);
            }
        }
        return result;
    }

    // Constructs dummy handler which will set specifier status code
    // and produce response with specified body and headers.
    // Please note that body can be null
    public abstract T dummyHandler(
        final int statusCode,
        final String body);

    public abstract T dummyHandler(
        final int statusCode,
        final String body,
        final Header header);

    public abstract T dummyHandler(
        final int statusCode,
        final String body,
        final Header... headers);

    // CSOFF: ParameterNumber
    private static <T> void addHandler(
        final String method,
        final String uri,
        final T handler,
        final Map<String, Map<T, List<String>>> handlers)
    {
        Map<T, List<String>> handlerMap = handlers.get(uri);
        if (handlerMap == null) {
            handlerMap = new LinkedHashMap<>();
            handlers.put(uri, handlerMap);
        }
        List<String> methods = handlerMap.get(handler);
        if (methods == null) {
            methods = new ArrayList<>();
            handlerMap.put(handler, methods);
        }
        methods.add(method);
    }
    // CSON: ParameterNumber

    // Retuns map from handler name to map from handler object to methods list
    public Map<String, Map<T, List<String>>> handlersInfo() {
        Map<String, Map<T, List<String>>> handlers = new TreeMap<>();
        Set<Map.Entry<String, PatternMap<RequestInfo, T>>> matchersSet =
            matchers.entrySet();
        for (Map.Entry<String, PatternMap<RequestInfo, T>> entry
            : matchersSet)
        {
            entry.getValue().traverse(
                (pattern, handler) -> addHandler(
                    entry.getKey(),
                    pattern.toString(),
                    handler,
                    handlers));
        }
        for (Map.Entry<String, PatternMap<RequestInfo, T>> entry
            : matchersSet)
        {
            T asteriskHandler = entry.getValue().asterisk();
            if (asteriskHandler != null) {
                addHandler(
                    entry.getKey(),
                    ASTERISK,
                    asteriskHandler,
                    handlers);
            }
        }
        return handlers;
    }

    private static void append(
        final StringBuilder sb,
        final Iterable<String> elems)
    {
        Iterator<String> iter = elems.iterator();
        sb.append(iter.next());
        while (iter.hasNext()) {
            sb.append(',');
            sb.append(' ');
            sb.append(iter.next());
        }
    }

    public T globalOptionsHandler() {
        Map<String, Map<T, List<String>>> handlers = handlersInfo();
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, Map<T, List<String>>> entry
            : handlers.entrySet())
        {
            sb.append(entry.getKey());
            sb.append('\n');
            for (Map.Entry<T, List<String>> handler
                : entry.getValue().entrySet())
            {
                sb.append('\t');
                append(sb, handler.getValue());
                sb.append('\n');
                sb.append('\t');
                sb.append('\t');
                sb.append(handler.getKey());
                sb.append('\n');
            }
        }
        return dummyHandler(HttpStatus.SC_OK, sb.toString());
    }

    public Set<String> allowedMethods(final RequestInfo request) {
        Set<String> methods = new LinkedHashSet<>();
        for (Map.Entry<String, PatternMap<RequestInfo, T>> entry
            : matchers.entrySet())
        {
            if (entry.getValue().get(request) != null) {
                methods.add(entry.getKey());
            }
        }
        return methods;
    }

    private static Header toAllowHeader(final Set<String> methods) {
        methods.add(OPTIONS);
        StringBuilder sb = new StringBuilder();
        append(sb, methods);
        return new BasicHeader(HttpHeaders.ALLOW, sb.toString());
    }

    public T optionsHandler(final RequestInfo request) {
        Set<String> methods = allowedMethods(request);
        if (methods.isEmpty()) {
            return null;
        } else {
            return dummyHandler(
                HttpStatus.SC_OK,
                null,
                toAllowHeader(methods));
        }
    }

    public T notFoundHandler(final RequestInfo request) {
        Set<String> methods = allowedMethods(request);
        if (methods.isEmpty()) {
            return null;
        } else {
            return dummyHandler(
                HttpStatus.SC_METHOD_NOT_ALLOWED,
                null,
                toAllowHeader(methods));
        }
    }
}

