package ru.yandex.http.util.request;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpRequest;

import ru.yandex.collection.PatternSample;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.parser.uri.UriParser;

public class RequestInfo implements PatternSample {
    private static final CollectionParser<
        String,
        List<String>,
        RuntimeException> OUTER_PARSER = new CollectionParser<>(
            GenericFunction.identity(),
            ArrayList::new,
            ';');
    private static final CollectionParser<
        String,
        List<String>,
        RuntimeException> INNER_PARSER =
            new CollectionParser<>(String::trim, ArrayList::new);

    private final HeadersParser headers;
    private final String method;
    private final List<String> paths;
    private final ScanningCgiParams params;
    private final Set<String> experiments;

    public RequestInfo(final HttpRequest request) {
        this(
            new HeadersParser(request),
            request.getRequestLine().getMethod(),
            request.getRequestLine().getUri());
    }

    public RequestInfo(
        final HeadersParser headers,
        final String method,
        final String uri)
    {
        this(headers, method, new UriParser(uri));
    }

    public RequestInfo(
        final HeadersParser headers,
        final String method,
        final UriParser uri)
    {
        this.headers = headers;
        this.method = method;
        paths = parseUri(uri.rawPath());
        params = new ScanningCgiParams(uri.queryParser());
        String experiments = headers.getString(
            YandexHeaders.X_ENABLED_BOXES,
            "");
        if (experiments.isEmpty()) {
            this.experiments = Collections.emptySet();
        } else {
            this.experiments = new HashSet<>();
            for (String experiment: OUTER_PARSER.apply(experiments)) {
                List<String> experimentBox = INNER_PARSER.apply(experiment);
                if (experimentBox.size() == 2 + 1) {
                    this.experiments.add(experimentBox.get(0));
                }
            }
        }
    }

    private static List<String> parseUri(final String uri) {
        if (uri.length() > 0 && uri.charAt(0) != '/' && uri.charAt(0) != '*') {
            //skip schema
            int schemaEnd = uri.indexOf("://");
            if (schemaEnd != -1) {
                schemaEnd += 1 + 2;
                int pathStart = uri.indexOf('/', schemaEnd);
                String pathWithPort = null;
                int i;
                if (pathStart == -1) {
                    i = uri.length();
                } else {
                    i = pathStart;
                }
                while (i-- > 0) {
                    char c = uri.charAt(i);
                    if (c > ':' || c < '0') {
                        break;
                    } else if (c == ':') {
                        pathWithPort = uri.substring(i);
                        break;
                    }
                }
                List<String> paths;
                if (pathStart == -1) {
                    if (pathWithPort == null) {
                        paths = Collections.singletonList(uri);
                    } else {
                        paths = Arrays.asList(uri, pathWithPort);
                    }
                } else {
                    String path = uri.substring(pathStart);
                    if (pathWithPort == null) {
                        paths = Arrays.asList(uri, path);
                    } else {
                        paths = Arrays.asList(uri, pathWithPort, path);
                    }
                }
                return paths;
            }
        }
        return Collections.singletonList(uri);
    }

    @Override
    public List<String> paths() {
        return paths;
    }

    public String method() {
        return method;
    }

    public ScanningCgiParams params() {
        return params;
    }

    public HeadersParser headers() {
        return headers;
    }

    public Set<String> experiments() {
        return experiments;
    }
}

