package ru.yandex.http.util.request;

import java.io.IOException;
import java.text.ParseException;
import java.util.Iterator;
import java.util.List;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import ru.yandex.function.CorpUidPredicate;
import ru.yandex.function.EqualablePredicate;
import ru.yandex.function.GenericFunction;
import ru.yandex.parser.query.FieldQuery;
import ru.yandex.parser.query.PredicateQueryVisitor;
import ru.yandex.parser.query.QueryAtom;
import ru.yandex.parser.query.QueryParser;
import ru.yandex.parser.query.QueryToken;
import ru.yandex.parser.query.QuotedQuery;
import ru.yandex.parser.string.BooleanParser;
import ru.yandex.util.string.StringUtils;

public enum RequestInfoPredicateParser
    implements GenericFunction<String, Predicate<RequestInfo>, ParseException>,
        PredicateQueryVisitor<RequestInfo, ParseException>
{
    INSTANCE;

    private static final String ARG = "arg_";
    private static final String ARGONLY = "argonly_";
    private static final String ARGREGEX = "argregex_";
    private static final String HTTP = "http_";
    private static final String CORP = "corp_";
    private static final String NULL = "null";

    @Override
    public Predicate<RequestInfo> apply(final String expr)
        throws ParseException
    {
        try {
            return new QueryParser(expr).parse().accept(this);
        } catch (IOException e) {
            ParseException ex =
                new ParseException("Failed to parse query: " + expr, 0);
            ex.initCause(e);
            throw ex;
        }
    }

    @Override
    public Predicate<RequestInfo> visit(final FieldQuery query)
        throws ParseException
    {
        List<String> fields = query.fields();
        if (fields.size() != 1) {
            throw new ParseException("Multi-field queries not allowed", 0);
        }
        QueryAtom atom = query.query();
        String field = fields.get(0);
        Predicate<RequestInfo> predicate;
        if (field.startsWith(ARG)) {
            String arg = field.substring(ARG.length());
            predicate = atom.accept(new ArgQueryVisitor(arg));
        } else if (field.startsWith(ARGONLY)) {
            String arg = field.substring(ARGONLY.length());
            predicate = atom.accept(new ArgOnlyQueryVisitor(arg));
        } else if (field.startsWith(ARGREGEX)) {
            String arg = field.substring(ARGREGEX.length());
            predicate = atom.accept(new ArgRegexQueryVisitor(arg));
        } else if (field.startsWith(HTTP)) {
            String header = field.substring(HTTP.length());
            predicate = atom.accept(new HeaderQueryVisitor(header));
        } else if (field.startsWith(CORP)) {
            String arg = field.substring(CORP.length());
            predicate = atom.accept(new CorpQueryVisitor(arg));
        } else if (field.equals("method")) {
            predicate = atom.accept(MethodQueryVisitor.INSTANCE);
        } else if (field.equals("experiment")) {
            predicate = atom.accept(ExperimentQueryVisitor.INSTANCE);
        } else if (field.equals("path_regex")) {
            predicate = atom.accept(PathRegexQueryVisitor.INSTANCE);
        } else {
            throw new ParseException("Unknown field: " + field, 0);
        }
        return predicate;
    }

    @Override
    public Predicate<RequestInfo> visit(final QueryToken query)
        throws ParseException
    {
        throw new ParseException(
            "Bare tokens not supported " + query.text(),
            0);
    }

    @Override
    public Predicate<RequestInfo> visit(final QuotedQuery query)
        throws ParseException
    {
        throw new ParseException("Quoted expressions not allowed", 0);
    }

    private abstract static class RequestInfoQueryVisitor
        implements PredicateQueryVisitor<RequestInfo, ParseException>
    {
        @Override
        public Predicate<RequestInfo> visit(final FieldQuery query)
            throws ParseException
        {
            throw new ParseException("Nested fields not allowed", 0);
        }

        @Override
        public Predicate<RequestInfo> visit(final QueryToken query)
            throws ParseException
        {
            return predicateFor(query.text());
        }

        @Override
        public Predicate<RequestInfo> visit(final QuotedQuery query)
            throws ParseException
        {
            StringBuilder sb = new StringBuilder();
            for (QueryToken token: query.tokens()) {
                if (sb.length() > 0) {
                    sb.append(' ');
                }
                sb.append(token.text());
            }
            return predicateFor(new String(sb));
        }

        protected abstract Predicate<RequestInfo> predicateFor(String text)
            throws ParseException;
    }

    private static class ArgQueryVisitor extends RequestInfoQueryVisitor {
        private final String arg;

        ArgQueryVisitor(final String arg) {
            this.arg = arg;
        }

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text) {
            return new ArgPredicate(arg, text);
        }
    }

    private static class ArgPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String arg;
        private final String text;

        ArgPredicate(final String arg, final String text) {
            this.arg = arg;
            this.text = text;
        }

        @Override
        public boolean test(final RequestInfo info) {
            return info.params().getString(arg, NULL).equals(text);
        }

        @Override
        public int hashCode() {
            return ARG.hashCode() ^ arg.hashCode() ^ text.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof ArgPredicate) {
                ArgPredicate other = (ArgPredicate) o;
                return arg.equals(other.arg) && text.equals(other.text);
            }
            return false;
        }

        @Override
        public String toString() {
            return StringUtils.concat(ARG, arg, ':', text);
        }
    }

    private static class ArgOnlyQueryVisitor extends RequestInfoQueryVisitor {
        private final String arg;

        ArgOnlyQueryVisitor(final String arg) {
            this.arg = arg;
        }

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text) {
            return new ArgOnlyPredicate(arg, text);
        }
    }

    private static class ArgOnlyPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String arg;
        private final String text;

        ArgOnlyPredicate(final String arg, final String text) {
            this.arg = arg;
            this.text = text;
        }

        @Override
        public boolean test(final RequestInfo info) {
            Iterator<String> iter = info.params().getAllOrNull(arg);
            if (iter != null && iter.hasNext()) {
                String value = iter.next();
                if (!iter.hasNext()) {
                    return text.equals(value);
                }
            }
            return false;
        }

        @Override
        public int hashCode() {
            return ARGONLY.hashCode() ^ arg.hashCode() ^ text.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof ArgOnlyPredicate) {
                ArgOnlyPredicate other = (ArgOnlyPredicate) o;
                return arg.equals(other.arg) && text.equals(other.text);
            }
            return false;
        }

        @Override
        public String toString() {
            return StringUtils.concat(ARGONLY, arg, ':', text);
        }
    }

    private static class ArgRegexQueryVisitor extends RequestInfoQueryVisitor {
        private final String arg;

        ArgRegexQueryVisitor(final String arg) {
            this.arg = arg;
        }

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text)
            throws ParseException
        {
            return new ArgRegexPredicate(arg, text);
        }
    }

    private static class ArgRegexPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String arg;
        private final Pattern pattern;

        ArgRegexPredicate(final String arg, final String pattern)
            throws ParseException
        {
            this.arg = arg;
            try {
                this.pattern = Pattern.compile(pattern);
            } catch (RuntimeException e) {
                ParseException ex = new ParseException(
                    "Failed to parse pattern <" + pattern + '>',
                    0);
                ex.initCause(e);
                throw ex;
            }
        }

        @Override
        public boolean test(final RequestInfo info) {
            Iterator<String> iter = info.params().getAllOrNull(arg);
            if (iter != null && iter.hasNext()) {
                String value = iter.next();
                if (!iter.hasNext()) {
                    return pattern.matcher(value).matches();
                }
            }
            return false;
        }

        @Override
        public int hashCode() {
            return ARGREGEX.hashCode()
                ^ arg.hashCode()
                ^ pattern.pattern().hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof ArgRegexPredicate) {
                ArgRegexPredicate other = (ArgRegexPredicate) o;
                return arg.equals(other.arg)
                    && pattern.pattern().equals(other.pattern.pattern());
            }
            return false;
        }

        @Override
        public String toString() {
            return StringUtils.concat(ARGREGEX, arg, ':', pattern.pattern());
        }
    }

    private static class HeaderQueryVisitor extends RequestInfoQueryVisitor {
        private final String header;

        HeaderQueryVisitor(final String header) {
            this.header = header;
        }

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text) {
            return new HeaderPredicate(header, text);
        }
    }

    private static class HeaderPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String header;
        private final String text;

        HeaderPredicate(final String header, final String text) {
            this.header = header;
            this.text = text;
        }

        @Override
        public boolean test(final RequestInfo info) {
            return info.headers().getString(header, NULL).equals(text);
        }

        @Override
        public int hashCode() {
            return HTTP.hashCode() ^ header.hashCode() ^ text.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof HeaderPredicate) {
                HeaderPredicate other = (HeaderPredicate) o;
                return header.equals(other.header) && text.equals(other.text);
            }
            return false;
        }

        @Override
        public String toString() {
            return StringUtils.concat(HTTP, header, ':', text);
        }
    }

    private static class CorpQueryVisitor extends RequestInfoQueryVisitor {
        private final String arg;

        CorpQueryVisitor(final String arg) {
            this.arg = arg;
        }

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text)
            throws ParseException
        {
            boolean value;
            try {
                value = BooleanParser.INSTANCE.apply(text);
            } catch (Exception e) {
                ParseException ex = new ParseException(
                    "Failed to parse condition: " + text,
                    0);
                ex.initCause(e);
                throw ex;
            }
            return new CorpPredicate(arg, value);
        }
    }

    private static class CorpPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String arg;
        private final boolean expectedValue;

        CorpPredicate(final String arg, final boolean expectedValue) {
            this.arg = arg;
            this.expectedValue = expectedValue;
        }

        @Override
        public boolean test(final RequestInfo info) {
            long uid;
            try {
                uid = info.params().getLong(arg);
            } catch (Exception e) {
                return false;
            }
            return CorpUidPredicate.INSTANCE.test(uid) == expectedValue;
        }

        @Override
        public int hashCode() {
            return CORP.hashCode()
                ^ arg.hashCode()
                ^ Boolean.hashCode(expectedValue);
        }

        @Override
        public boolean equals(final Object o) {
            if (o instanceof CorpPredicate) {
                CorpPredicate other = (CorpPredicate) o;
                return arg.equals(other.arg)
                    && expectedValue == other.expectedValue;
            }
            return false;
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder(CORP);
            sb.append(arg);
            sb.append(':');
            sb.append(expectedValue);
            return new String(sb);
        }
    }

    private static class MethodQueryVisitor extends RequestInfoQueryVisitor {
        public static final MethodQueryVisitor INSTANCE =
            new MethodQueryVisitor();

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text) {
            return new MethodPredicate(text);
        }
    }

    private static class MethodPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String method;

        MethodPredicate(final String method) {
            this.method = method;
        }

        @Override
        public boolean test(final RequestInfo info) {
            return method.equals(info.method());
        }

        @Override
        public int hashCode() {
            return method.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof MethodPredicate
                && method.equals(((MethodPredicate) o).method);
        }

        @Override
        public String toString() {
            return StringUtils.concat("method:", method);
        }
    }

    private static class ExperimentQueryVisitor
        extends RequestInfoQueryVisitor
    {
        public static final ExperimentQueryVisitor INSTANCE =
            new ExperimentQueryVisitor();

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text) {
            return new ExperimentPredicate(text);
        }
    }

    private static class ExperimentPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final String experiment;

        ExperimentPredicate(final String experiment) {
            this.experiment = experiment;
        }

        @Override
        public boolean test(final RequestInfo info) {
            return info.experiments().contains(experiment);
        }

        @Override
        public int hashCode() {
            return experiment.hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof ExperimentPredicate
                && experiment.equals(((ExperimentPredicate) o).experiment);
        }

        @Override
        public String toString() {
            return StringUtils.concat("experiment:", experiment);
        }
    }

    private static class PathRegexQueryVisitor
        extends RequestInfoQueryVisitor
    {
        public static final PathRegexQueryVisitor INSTANCE =
            new PathRegexQueryVisitor();

        @Override
        protected Predicate<RequestInfo> predicateFor(final String text)
            throws ParseException
        {
            return new PathRegexPredicate(text);
        }
    }

    private static class PathRegexPredicate
        implements EqualablePredicate<RequestInfo>
    {
        private final Pattern pattern;

        PathRegexPredicate(final String pattern) throws ParseException {
            try {
                this.pattern = Pattern.compile(pattern);
            } catch (RuntimeException e) {
                ParseException ex = new ParseException(
                    "Failed to parse pattern <" + pattern + '>',
                    0);
                ex.initCause(e);
                throw ex;
            }
        }

        @Override
        public boolean test(final RequestInfo info) {
            return pattern.matcher(info.paths().get(0)).matches();
        }

        @Override
        public int hashCode() {
            return pattern.pattern().hashCode();
        }

        @Override
        public boolean equals(final Object o) {
            return o instanceof PathRegexPredicate
                && pattern.pattern().equals(
                    ((PathRegexPredicate) o).pattern.pattern());
        }

        @Override
        public String toString() {
            return StringUtils.concat("path_regex:", pattern.pattern());
        }
    }
}

