package ru.yandex.http.util.request.function;

import ru.yandex.function.ByteArrayCopier;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.Processable;
import ru.yandex.function.VoidProcessor;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.UnhexStrings;

public class BinaryValue
    extends ByteArrayProcessable
    implements RequestFunctionValue
{
    // buf object + (buf.length + padding) + off + len
    private static final int BASE_WEIGHT =
        OBJECT_WEIGHT + Long.BYTES + Integer.BYTES + Integer.BYTES;
    private static final int HASH_CODE_MULTIPLIER = 31;
    private static final int DUMP_LENGTH = 8;
    private static final String LONG_STRING_SUFFIX = "...";

    public BinaryValue(final Processable<byte[]> value) {
        super(value.processWith(ByteArrayCopier.INSTANCE));
    }

    @Override
    public int weight() {
        return BASE_WEIGHT + buf.length;
    }

    @Override
    public String stringValue() {
        return new String(processWith(HexStrings.UPPER));
    }

    @Override
    public <E extends Exception> void appendTo(
        final VoidProcessor<? super byte[], E> processor)
        throws E
    {
        processWith(processor);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("BinaryValue(length=");
        sb.append(len);
        sb.append(",data=");
        if (len <= DUMP_LENGTH + LONG_STRING_SUFFIX.length()) {
            HexStrings.UPPER.toStringBuilder(sb, buf, len);
        } else {
            HexStrings.UPPER.toStringBuilder(sb, buf, DUMP_LENGTH);
            sb.append(LONG_STRING_SUFFIX);
        }
        sb.append(')');
        return new String(sb);
    }

    @Override
    public int hashCode() {
        int result = 1;
        for (int i = 0; i < len; ++i) {
            result = HASH_CODE_MULTIPLIER * result + buf[i + off];
        }
        return result;
    }

    // CSOFF: ReturnCount
    private boolean equals(final byte[] buf, final int off, final int len) {
        if (this.len != len) {
            return false;
        }
        for (int i = 0; i < len; ++i) {
            if (this.buf[i + this.off] != buf[i + off]) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o instanceof BinaryValue) {
            BinaryValue other = (BinaryValue) o;
            return equals(other.buf, other.off, other.len);
        }
        return false;
    }
    // CSON: ReturnCount

    @Override
    public boolean equalsString(final String value) {
        try {
            byte[] buf = UnhexStrings.unhex(value);
            return equals(buf, 0, buf.length);
        } catch (RuntimeException e) {
            return false;
        }
    }
}

