package ru.yandex.http.util.request.function;

import java.io.IOException;
import java.io.StringReader;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import ru.yandex.function.GenericFunction;
import ru.yandex.parser.string.EnumParser;

public class RequestFunctionFactory<V extends RequestFunctionContext>
    implements GenericFunction<String, RequestFunction<V>, ParseException>
{
    public static final RequestFunctionFactory<
        RequestFunctionContext> INSTANCE =
            new RequestFunctionFactory<RequestFunctionContext>();

    protected RequestFunctionFactory() {
    }

    @Override
    public RequestFunction<V> apply(final String str) throws ParseException {
        int openBracket = str.indexOf('(');
        int closeBracket = str.lastIndexOf(')');
        if (openBracket == -1) {
            throw new ParseException(
                "Invalid function syntax:  " + str
                    + ", valid syntax is: functionName(optionalParameter1, "
                    + "optionalParameter2...)",
                0);
        }
        final String name = str.substring(0, openBracket);
        final String params = str.substring(openBracket + 1, closeBracket + 1);
        if (name.length() == 0) {
            throw new ParseException(
                "Invalid function syntax: " + str
                    + ": empty function name",
                0);
        }
        final RequestFunction<V>[] args = parse(params);
        return createFunction(name, args);
    }

    private RequestFunction<V>[] parse(final String str)
        throws ParseException
    {
        try {
            return parse(new StringReader(str));
        } catch (IOException e) {
            ParseException pe = new ParseException("Unhandled IOException", 0);
            pe.addSuppressed(e);
            throw pe;
        }
    }

    private RequestFunction<V>[] parse(final StringReader reader)
        throws IOException, ParseException
    {
        boolean escape = false;
        List<RequestFunction<V>> args = new ArrayList<>();
        final StringBuilder sb = new StringBuilder();
        int c;
        boolean expectArg = false;
        boolean skipComma = false;
        while ((c = reader.read()) != -1) {
            if (c == '\\' && !escape) {
                escape = true;
                continue;
            }
            if (!escape) {
                if (c == '(') {
                    final String name = new String(sb);
                    sb.setLength(0);
                    try {
                        RequestFunction<V>[] subArgs = parse(reader);
                        args.add(createFunction(name, subArgs));
                    } catch (ParseException e) {
                        throw new ParseException(
                            "Error parsing function <"
                            + name + ">: " + e.getMessage(),
                            0);
                    }
                    // There will be comma after function invocation, like
                    // concat(get(uid),suffix), so comma expected after it
                    skipComma = true;
                } else if (c == ',') {
                    if (skipComma) {
                        skipComma = false;
                    } else {
                        args.add(createConst(new String(sb)));
                        sb.setLength(0);
                        expectArg = true;
                    }
                } else if (c == ')') {
                    if (expectArg || sb.length() > 0) {
                        args.add(createConst(new String(sb)));
                        expectArg = false;
                        sb.setLength(0);
                    }
                    break;
                } else {
                    sb.append((char) c);
                    expectArg = false;
                }
            } else {
                escape = false;
                sb.append((char) c);
            }
        }
        if (sb.length() > 0) {
            throw new ParseException(
                "Unexpected expression end, forgotten ')' after '" + sb + "'?",
                0);
        }
        @SuppressWarnings({"unchecked", "rawtypes"})
        RequestFunction<V>[] argsArray =
            new RequestFunction[args.size()];
        return args.toArray(argsArray);
    }

    private RequestFunction<V> createConst(final String constStr) {
        return new ConstFunction<V>(constStr);
    }

    protected RequestFunction<V> createFunction(
        final String name,
        final RequestFunction<V>[] args)
        throws ParseException
    {
        try {
            return new EnumParser<RequestFunctionType>(
                RequestFunctionType.class)
                    .apply(name)
                    .create(args);
        } catch (RuntimeException e) {
            throw new ParseException(
                "Failed to parse function: " + name, 0);
        }
    }
}
