package ru.yandex.http.util.request.function;

import java.net.MalformedURLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ExecutionException;

import org.apache.http.HttpHost;

import ru.yandex.http.util.HttpHostParser;
import ru.yandex.parser.string.CollectionParser;

public class StringValue implements RequestFunctionValue {
    // char array + (array.length + padding) + off + len + hash
    private static final int BASE_WEIGHT =
        OBJECT_WEIGHT + Long.BYTES
        + Integer.BYTES + Integer.BYTES + Integer.BYTES;
    private static final CollectionParser<
        HttpHost,
        List<HttpHost>,
        MalformedURLException> HOST_LIST_PARSER =
            new CollectionParser<>(HttpHostParser.INSTANCE, ArrayList::new);

    private final String value;

    public StringValue(final String value) {
        this.value = value;
    }

    public StringValue(final RequestFunctionValue value) {
        this.value = value.stringValue();
    }

    @Override
    public int weight() {
        return BASE_WEIGHT + (value.length() << 1);
    }

    @Override
    public String stringValue() {
        return value;
    }

    @Override
    public long longValue() throws ExecutionException {
        try {
            return Long.parseLong(value);
        } catch (NumberFormatException e) {
            throw new ExecutionException("String  <" + value + "> cannot"
                + " be converted to long", e);
        }
    }

    @Override
    public double doubleValue() throws ExecutionException {
        try {
            return Double.parseDouble(value);
        } catch (NumberFormatException e) {
            throw new ExecutionException("String <" + value + "> cannot "
                + "be converted to double", e);
        }
    }

    @Override
    public boolean booleanValue() {
        return value != null && value.length() > 0;
    }

    @Override
    public List<HttpHost> hostListValue() throws ExecutionException {
        try {
            return HOST_LIST_PARSER.apply(value);
        } catch (Exception e) {
            throw new ExecutionException(
                "Failed to parse '" + value + "' as host list",
                e);
        }
    }

    @Override
    public String toString() {
        return "StringValue(" + value + ')';
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(value);
    }

    @Override
    public boolean equals(final Object o) {
        return this == o
            || (o instanceof StringValue
                && Objects.equals(value, ((StringValue) o).value));
    }

    @Override
    public boolean equalsString(final String value) {
        return Objects.equals(this.value, value);
    }
}
