package ru.yandex.http.server.sync;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.net.StandardSocketOptions;
import java.nio.channels.CancelledKeyException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.SSLEngine;

import org.apache.http.ConnectionClosedException;
import org.apache.http.HttpException;
import org.apache.http.MalformedChunkCodingException;
import org.apache.http.TruncatedChunkException;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.protocol.HttpCoreContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.collection.BlockingBlockingQueue;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.util.SynchronizedHttpContext;
import ru.yandex.http.util.request.function.InetAddressValue;
import ru.yandex.http.util.server.AbstractHttpServer;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.http.util.server.Limiter;
import ru.yandex.http.util.server.LimiterResult;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.timesource.TimeSource;

public class BaseHttpServer<T extends ImmutableBaseServerConfig>
    extends AbstractHttpServer<T, HttpRequestHandler>
{
    private static final String CONNECTION_RESET_BY_PEER =
        "Connection reset by peer";
    private static final String MS = " ms";
    private static final String CONNECTION = "Connection ";

    @SuppressWarnings("JdkObsolete")
    private final List<LoggingHttpServerConnection> connections =
        new LinkedList<>();
    private final Queue<LoggingHttpServerConnection> newConnections =
        new ConcurrentLinkedQueue<>();
    private final Queue<LoggingHttpServerConnection> queuedConnections =
        new ConcurrentLinkedQueue<>();
    private final ConcurrentLinkedQueue<
        LoggingHttpServerConnection> closeWaitQueue =
            new ConcurrentLinkedQueue<>();
    private final LinkedBlockingQueue<LoggingHttpServerConnection> closeQueue =
        new LinkedBlockingQueue<>();
    private final Selector selector = Selector.open();
    private final long timerResolution;
    private final int maxGarbageConnections;
    private final long socketTimeout;
    private final BaseHttpService httpService;
    private final ThreadPoolExecutor threadPool;
    private final CloseQueueProcessor closer;

    private volatile boolean bound = false;
    private volatile boolean stopped = false;
    private Listener listener;
    private Listener httpListener;
    private long lastCleanupTime = TimeSource.INSTANCE.currentTimeMillis();
    private int addedSinceLastCleanup = 0;

    public BaseHttpServer(final T config)
        throws IOException
    {
        this(config, new CompositeHttpRequestHandlerMapper());
    }

    public BaseHttpServer(
        final T config,
        final CompositeHttpRequestHandlerMapper handlerMapper)
        throws IOException
    {
        super(config, handlerMapper);
        timerResolution = config.timerResolution();
        maxGarbageConnections = config.maxGarbageConnections();
        socketTimeout = config.timeout();
        httpService = new BaseHttpService(
            handlerMapper,
            serviceContextRenewalTask,
            this,
            () -> pingEnabled(null));

        threadPool = new ThreadPoolExecutor(
            config.workers(),
            config.workers(),
            1,
            TimeUnit.MINUTES,
            new BlockingBlockingQueue<>(
                new LifoWaitBlockingQueue<>(config.connections())),
            new NamedThreadFactory(
                getThreadGroup(),
                getName() + '-'));
        closeChain.add(new ThreadPoolCloser());
        closer = new CloseQueueProcessor(this);
        closeChain.add(closer);
        if (!config.lazyBind()) {
            bind();
            bound = true;
        }
        closeChain.add(new ServerCloser());

        register(
            new Pattern<>("/ping", false),
            new PingHandler(this),
            false);
        register(
            new Pattern<>("/disablePing", false),
            new DisablePingHandler(this::disablePing),
            false);
        register(
            new Pattern<>("/disable-ping", false),
            new DisablePingHandler(this::disablePing),
            false);
        register(
            new Pattern<>("/enable-ping", false),
            new EnablePingHandler(this::enablePing),
            false);
        register(
            new Pattern<>("/config-update", false),
            new DynamycConfigUpdateHandler(this),
            false);
        register(
            new Pattern<>("/logrotate", false),
            new LogRotateHandler(this),
            false);
        register(
            new Pattern<>("/status", false),
            new StatusHandler(this),
            false);
        register(
            new Pattern<>("/stat", false),
            new StatHandler(this),
            false);
        register(
            new Pattern<>("/systemexit", false),
            new SystemExitHandler(),
            false);
        register(
            new Pattern<>("/force-gc", false),
            new ForceGcHandler(),
            false);

        register(
            new Pattern<>("/add-debug-flag", false),
            new AddDebugFlagHandler(this),
            false);
        register(
            new Pattern<>("/remove-debug-flag", false),
            new RemoveDebugFlagHandler(this),
            false);
        register(
            new Pattern<>("/debug-flags", false),
            new DebugFlagsHandler(this),
            false);

        if (ibmDumper != null) {
            DumpHandler dumpHandler = new DumpHandler(ibmDumper);
            register(new Pattern<>("/javadump", false), dumpHandler, false);
            register(new Pattern<>("/systemdump", false), dumpHandler, false);
            register(new Pattern<>("/heapdump", false), dumpHandler, false);
        }

        register(
            new Pattern<>("/custom-golovan-panel", false),
            new CustomGolovanPanelHandler(this),
            false);
        register(
            new Pattern<>("/custom-alerts-config", false),
            new CustomAlertsConfigHandler(this),
            false);
        if (golovanPanelConfig != null) {
            register(
                new Pattern<>("/generate-golovan-panel", false),
                new GenerateGolovanPanelHandler(this),
                false);
            ImmutableGolovanAlertsConfig alerts = golovanPanelConfig.alerts();
            if (alerts != null) {
                register(
                    new Pattern<>("/generate-alerts-config", false),
                    new GenerateAlertsConfigHandler(this),
                    false);
            }
        }

        registerStater(new BaseHttpServerStater(threadPool));
    }

    private void bind() throws IOException {
        ConnectionConfig connConfig = config.toConnectionConfig();
        int port = config.port();
        if (httpsConfig == null) {
            listener = new Listener(
                this,
                new LoggingHttpServerConnectionFactory(this, connConfig),
                port);
            closeChain.add(listener);
            httpListener = listener;
        } else {
            listener = new Listener(
                this,
                new SSLLoggingHttpServerConnectionFactory(this, connConfig),
                port);
            closeChain.add(listener);
            int httpPort = httpsConfig.httpPort();
            if (httpPort >= 0) {
                httpListener = new Listener(
                    this,
                    new LoggingHttpServerConnectionFactory(this, connConfig),
                    httpPort);
                closeChain.add(httpListener);
            } else {
                httpListener = listener;
            }
        }
    }

    private void ensureBound() throws IOException {
        if (!bound) {
            synchronized (this) {
                if (!bound) {
                    bind();
                    bound = true;
                }
            }
        }
    }

    @Override
    protected HttpRequestHandler wrap(final HttpRequestHandler handler) {
        return new OverridableHttpRequestHandler(handler);
    }

    @Override
    protected HttpRequestHandler unwrap(final HttpRequestHandler handler) {
        if (handler instanceof OverridableHttpRequestHandler) {
            return ((OverridableHttpRequestHandler) handler).handler();
        } else {
            return handler;
        }
    }

    @Override
    public InetSocketAddress address() throws IOException {
        ensureBound();
        return listener.address();
    }

    @Override
    public InetSocketAddress httpAddress() throws IOException {
        ensureBound();
        return httpListener.address();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        status.put(QUEUED_CONNECTIONS, threadPool.getQueue().size());
        status.put(ACTIVE_WORKERS, threadPool.getActiveCount());
        status.put(SPAWNED_WORKERS, threadPool.getPoolSize());
        return status;
    }

    @Override
    public void start() throws IOException {
        closer.start();
        threadPool.prestartAllCoreThreads();
        super.start();
        listener.start();
        if (listener != httpListener) {
            httpListener.start();
        }
    }

    public SSLEngine createSSLEngine() {
        SSLEngine sslEngine = sslContext.createSSLEngine();
        config.httpsConfig().initialize(sslEngine);
        return sslEngine;
    }

    private void processSelectedKeys() throws IOException {
        Set<SelectionKey> selected = selector.selectedKeys();
        for (SelectionKey key: selected) {
            handleNotSoNewConnection(key);
        }
        selected.clear();
    }

    private boolean idlefy(final LoggingHttpServerConnection conn) {
        try {
            conn.idlefy(selector);
        } catch (IOException | CancelledKeyException e) {
            try {
                conn.shutdown();
            } catch (IOException ex) {
                e.addSuppressed(ex);
            }
            logger.log(Level.WARNING, "Failed to reregister connection", e);
            return false;
        }
        return true;
    }

    @Override
    public void run() {
        while (!stopped) {
            try {
                if (selector.select(timerResolution) > 0) {
                    processCloseWait();
                    processSelectedKeys();
                }
                while (selector.selectNow() > 0) {
                    processCloseWait();
                    processSelectedKeys();
                }
                processCloseWait();
                long now = TimeSource.INSTANCE.currentTimeMillis();
                boolean forceCleanup = false;
                boolean cleanupRequired =
                    now - lastCleanupTime >= timerResolution
                    || addedSinceLastCleanup >= maxGarbageConnections;
                LoggingHttpServerConnection conn = newConnections.poll();
                while (conn != null) {
                    if (conn == LoggingHttpServerConnection.DEAD) {
                        forceCleanup = true;
                    } else {
                        if (idlefy(conn)) {
                            connections.add(conn);
                            ++addedSinceLastCleanup;
                        } else {
                            cleanupRequired = true;
                        }
                    }
                    conn = newConnections.poll();
                }
                conn = queuedConnections.poll();
                if (forceCleanup) {
                    int cleaned = 0;
                    while (conn != null) {
                        if (conn.inAir()) {
                            // This connection just done SSL handshake and
                            // waiting for input, don't kill it, please
                            idlefy(conn);
                        } else {
                            ++cleaned;
                            // Sorry, to many alive connections, time to kill
                            try {
                                conn.shutdown();
                            } catch (IOException e) {
                                logger.log(
                                    Level.WARNING,
                                    "Failed to close reregistered connection",
                                    e);
                            }
                        }
                        conn = queuedConnections.poll();
                    }
                    if (logger.isLoggable(Level.INFO)) {
                        logger.info("Purged connections count: " + cleaned);
                    }
                } else {
                    while (conn != null) {
                        if (!idlefy(conn)) {
                            cleanupRequired = true;
                        }
                        conn = queuedConnections.poll();
                    }
                }
                if (cleanupRequired || forceCleanup) {
                    cleanupConnections(now);
                }
            } catch (ClosedChannelException e) {
                if (!stopped) {
                    logger.info("Channel closed");
                }
            } catch (IOException e) {
                if (!stopped) {
                    logger.log(Level.SEVERE, "Accept failed", e);
                }
            } catch (Throwable t) {
                if (!stopped) {
                    logger.log(Level.SEVERE, "Failed to accept connection", t);
                }
            }
        }
        logger.info("Request processing stopped");
    }

    private void processCloseWait() {
        Iterator<LoggingHttpServerConnection> iter = closeWaitQueue.iterator();
        while (iter.hasNext()) {
            LoggingHttpServerConnection conn = iter.next();
            if (!conn.channel().isRegistered()) {
                closeQueue.add(conn);
                iter.remove();
            }
        }
    }

    public void defferClose(final LoggingHttpServerConnection conn) {
        closeWaitQueue.add(conn);
    }

    private void cleanupConnections(final long now) {
        int cleaned = 0;
        Iterator<LoggingHttpServerConnection> iter = connections.iterator();
        while (iter.hasNext()) {
            LoggingHttpServerConnection conn = iter.next();
            boolean remove;
            long idleSince = conn.idleSince();
            if (idleSince == 0L) {
                Socket socket = conn.getSocket();
                remove = socket == null || socket.isClosed();
            } else {
                remove = now - idleSince > socketTimeout;
            }
            if (remove) {
                conn.selectionKey().cancel();
                try {
                    conn.shutdown();
                } catch (IOException e) {
                }
                iter.remove();
                ++cleaned;
            }
        }
        addedSinceLastCleanup = 0;
        lastCleanupTime = TimeSource.INSTANCE.currentTimeMillis();
        if ((cleaned > 0 || connections.size() > maxGarbageConnections)
            && logger.isLoggable(Level.FINE))
        {
            logger.fine(
                "Connections clean up completed, cleaned: "
                + cleaned + '/' + connections.size()
                + ", time taken: " + (lastCleanupTime - now) + MS);
        }
    }

    private void handleNotSoNewConnection(final SelectionKey key) {
        LoggingHttpServerConnection conn =
            (LoggingHttpServerConnection) key.attachment();
        key.cancel();
        if (conn == null) {
            logger.severe("Connection is null for " + key);
        } else {
            boolean failed = true;
            IOException ex = null;
            try {
                conn.requestStarted();
                conn.busyfy();
                try {
                    threadPool.execute(new Worker(conn));
                    failed = false;
                } catch (RejectedExecutionException e) {
                    if (!stopped) {
                        logger.log(
                            Level.WARNING,
                            "Failed to process connection: " + conn,
                            e);
                    }
                    try {
                        conn.shutdown();
                    } catch (IOException exc) {
                        ex = exc;
                        logger.log(
                            Level.FINER,
                            "Shutdown failed for " + conn,
                            exc);
                    }
                }
            } catch (IOException e) {
                ex = e;
            } finally {
                if (failed) {
                    closeFailedConnection(conn, ex);
                }
            }
        }
    }

    private void closeFailedConnection(
        final LoggingHttpServerConnection conn,
        final Exception e)
    {
        logger.log(
            Level.WARNING,
            "Failed to reinitialize connection object",
            e);
        try {
            conn.shutdown();
        } catch (IOException ex) {
            logger.log(Level.FINER, "Connection close failed", ex);
        }
    }

    private void reregisterConnection(final LoggingHttpServerConnection conn) {
        queuedConnections.add(conn);
        selector.wakeup();
    }

    private class Worker implements Runnable {
        private final LoggingHttpServerConnection connection;

        Worker(final LoggingHttpServerConnection connection) {
            this.connection = connection;
        }

        @Override
        public void run() {
            boolean failed = true;
            try {
                HttpCoreContext context =
                    new HttpCoreContext(new SynchronizedHttpContext());
                for (failed = false;
                    !failed && !threadPool.isShutdown();)
                {
                    Logger logger = BaseHttpServer.this.logger;
                    failed = true;
                    try {
                        if (logger.isLoggable(Level.FINE)) {
                            logger.fine("Queue wait time: "
                                + (TimeSource.INSTANCE.currentTimeMillis()
                                    - connection.requestStartTime()));
                        }
                        if (connection.needReregister()) {
                            failed = false;
                            if (logger.isLoggable(Level.INFO)) {
                                logger.info(
                                    CONNECTION + connection
                                    + " requested reregistration");
                            }
                            connection.setInAir();
                            reregisterConnection(connection);
                            break;
                        }
                        try {
                            httpService.handleRequest(connection, context);
                        } finally {
                            // logger can be altered by httpService
                            Logger contextLogger =
                                (Logger) context.getAttribute(LOGGER);
                            if (contextLogger != null) {
                                logger = contextLogger;
                            }
                        }
                        logger.info("Request processed");
                        if (connection.isOpen()) {
                            failed = false;
                            if (connection.hasBufferedData()) {
                                connection.requestStarted();
                            } else {
                                reregisterConnection(connection);
                                break;
                            }
                        } else {
                            break;
                        }
                    } catch (ConnectionClosedException e) {
                        logger.warning("Connection input closed unexpectedly");
                    } catch (TruncatedChunkException e) {
                        logger.log(
                            Level.WARNING,
                            "Connection input closed on incomplete chunk",
                            e);
                    } catch (MalformedChunkCodingException e) {
                        logger.log(
                            Level.WARNING,
                            "Connection input closed on malformed chunk",
                            e);
                    } catch (ClosedChannelException e) {
                        logger.finest("Client closed connection");
                    } catch (SocketTimeoutException e) {
                        logger.log(Level.SEVERE, "Socket timeout", e);
                    } catch (SocketException e) {
                        logger.log(Level.SEVERE, "Socket error", e);
                    } catch (IOException e) {
                        if (CONNECTION_RESET_BY_PEER.equals(e.getMessage())) {
                            logger.warning(CONNECTION_RESET_BY_PEER);
                        } else {
                            logger.log(Level.SEVERE, "I/O error", e);
                        }
                    } catch (HttpException e) {
                        logger.log(
                            Level.SEVERE,
                            "Unrecoverable HTTP protocol violation",
                            e);
                    } catch (Throwable e) {
                        logger.log(Level.SEVERE, "Error occured", e);
                    }
                }
                logger.info("Worker stopped");
            } finally {
                try {
                    if (failed) {
                        // Close connection only if error has been occured
                        // Otherwise connection should be reregistered
                        connection.shutdown();
                    }
                } catch (IOException e) {
                    logger.log(Level.INFO, "Failed to close connection", e);
                }
            }
        }
    }

    private static class Listener
        extends Thread
        implements GenericAutoCloseable<IOException>
    {
        private final ServerSocketChannel channel = ServerSocketChannel.open();
        private final BaseHttpServer<?> server;
        private final HttpServerConnectionFactory connectionFactory;
        private final long timerResolution;
        private final int socketTimeout;
        private final int linger;
        private final Limiter connectionsLimiter;
        private final boolean perKeyLimitsEnabled;
        private final boolean rejectConnectionsOverLimit;
        private final Logger logger;
        private final InetSocketAddress address;
        private volatile boolean closed = false;

        Listener(
            final BaseHttpServer<?> server,
            final HttpServerConnectionFactory connectionFactory,
            final int port)
            throws IOException
        {
            super(
                server.getThreadGroup(),
                server.getName() + '-' + port + "-Listener");
            this.server = server;
            this.connectionFactory = connectionFactory;
            ImmutableBaseServerConfig config = server.config;
            timerResolution = config.timerResolution();
            socketTimeout = config.timeout();
            linger = config.linger();
            connectionsLimiter = config.connectionsLimiter();
            perKeyLimitsEnabled = connectionsLimiter.perKeyLimitsEnabled();
            rejectConnectionsOverLimit = config.rejectConnectionsOverLimit();
            logger = server.logger();
            channel.bind(new InetSocketAddress(port), config.backlog());
            address = (InetSocketAddress) channel.getLocalAddress();
        }

        public InetSocketAddress address() {
            return address;
        }

        private void handle(final SocketChannel channel) throws IOException {
            try (GenericAutoCloseableHolder<
                    IOException,
                    GenericCloseableAdapter<SocketChannel>> channelHolder =
                    new GenericAutoCloseableHolder<>(
                        new GenericCloseableAdapter<>(channel)))
            {
                server.onIncomingConnection(channel);
                InetAddressValue limiterKey = null;
                if (perKeyLimitsEnabled) {
                    limiterKey =
                        new InetAddressValue(
                            ((InetSocketAddress) channel.getRemoteAddress())
                                .getAddress());
                }
                GenericAutoCloseable<RuntimeException> resourcesReleaser =
                    null;
                while (true) {
                    LimiterResult limiterResult =
                        connectionsLimiter.acquire(-1L, limiterKey);
                    String limiterMessage = limiterResult.message();
                    GenericAutoCloseable<RuntimeException> releaser =
                        limiterResult.resourcesReleaser();
                    if (limiterMessage == null) {
                        resourcesReleaser = releaser;
                        break;
                    } else {
                        if (releaser != null) {
                            releaser.close();
                        }
                        server.onLimitedConnection(channel, limiterMessage);
                        server.newConnections.add(
                            LoggingHttpServerConnection.DEAD);
                        server.selector.wakeup();
                        if (rejectConnectionsOverLimit) {
                            server.discardChannel(channel);
                            return;
                        }
                    }
                    try {
                        Thread.sleep(timerResolution);
                    } catch (InterruptedException e) {
                        server.discardChannel(channel);
                        Thread.currentThread().interrupt();
                        return;
                    }
                }
                try (GenericAutoCloseableHolder<
                        IOException,
                        LoggingHttpServerConnection> connHolder =
                        new GenericAutoCloseableHolder<>(
                            connectionFactory.create(
                                channel,
                                resourcesReleaser)))
                {
                    channelHolder.release();
                    channel.setOption(StandardSocketOptions.TCP_NODELAY, true);
                    channel.setOption(
                        StandardSocketOptions.SO_REUSEADDR,
                        true);
                    channel.setOption(StandardSocketOptions.SO_LINGER, linger);

                    LoggingHttpServerConnection conn = connHolder.get();
                    conn.setSocketTimeout(socketTimeout);
                    server.newConnections.add(conn);
                    server.selector.wakeup();
                    connHolder.release();
                }
            }
        }

        @Override
        public void run() {
            while (!closed) {
                try {
                    handle(channel.accept());
                } catch (IOException e) {
                    if (closed) {
                        logger.info("Listener closed");
                    } else {
                        logger.log(
                            Level.WARNING,
                            "Failed to accept new connection",
                            e);
                    }
                } catch (Exception e) {
                    logger.log(Level.SEVERE, "Unexpected listener thread exception", e);
                    e.printStackTrace();
                    throw e;
                }
            }
        }

        @Override
        public void close() throws IOException {
            closed = true;
            try {
                channel.close();
            } finally {
                interrupt();
            }
        }
    }

    private static class CloseQueueProcessor
        extends Thread
        implements GenericAutoCloseable<IOException>
    {
        private final LinkedBlockingQueue<
            LoggingHttpServerConnection> closeQueue;
        private final Logger logger;

        CloseQueueProcessor(
            final BaseHttpServer<?> server)
        {
            super(
                server.getThreadGroup(),
                server.getName() + "CloseQueueProcessor");
            closeQueue = server.closeQueue;
            logger = server.logger();
        }

        @Override
        public void run() {
            while (true) {
                try {
                    LoggingHttpServerConnection conn = closeQueue.take();
                    if (conn == LoggingHttpServerConnection.DEAD) {
                        break;
                    }
                    try {
                        conn.socketClose();
                    } catch (IOException e) {
                        logger.log(Level.SEVERE, "Error closing socket for "
                            + "connection: " + conn, e);
                    }
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.log(Level.SEVERE, "Interrupted", e);
                    break;
                }
            }
        }

        @Override
        public void close() {
            closeQueue.add(LoggingHttpServerConnection.DEAD);
            try {
                join();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private static class BaseHttpServerStater implements Stater {
        private final ThreadPoolExecutor threadPool;

        BaseHttpServerStater(final ThreadPoolExecutor threadPool) {
            this.threadPool = threadPool;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int queuedConnections = threadPool.getQueue().size();
            statsConsumer.stat("queued-connections_axxx", queuedConnections);
            statsConsumer.stat("queued-connections_ammm", queuedConnections);
            int activeWorkers = threadPool.getActiveCount();
            statsConsumer.stat("active-workers_axxx", activeWorkers);
            statsConsumer.stat("active-workers_ammm", activeWorkers);
            int spawnedWorkers = threadPool.getPoolSize();
            statsConsumer.stat("spawned-workers_axxx", spawnedWorkers);
            statsConsumer.stat("spawned-workers_ammm", spawnedWorkers);
        }
    }

    private class ThreadPoolCloser
        implements GenericAutoCloseable<IOException>
    {
        @Override
        public void close() throws IOException {
            try {
                threadPool.shutdown();
                for (LoggingHttpServerConnection conn: connections) {
                    try {
                        conn.shutdown();
                    } catch (IOException e) {
                        logger.log(
                            Level.FINER,
                            getName() + " failed to close connection: " + conn,
                            e);
                    }
                }
                while (!threadPool.awaitTermination(1, TimeUnit.SECONDS)) {
                    processCloseWait();
                    logger.warning(
                        "Some " + getName() + " threads still alive");
                }
                threadPool.shutdownNow();
                processCloseWait();
            } catch (InterruptedException e) {
                logger.log(
                    Level.WARNING,
                    getName() + " thread pool shutdown interrupted",
                    e);
                throw new IOException(e);
            }
        }
    }

    private class ServerCloser implements GenericAutoCloseable<IOException> {
        @Override
        public void close() throws IOException {
            try {
                stopped = true;
                interrupt();
                join();
                selector.close();
            } catch (InterruptedException e) {
                logger.log(
                    Level.WARNING,
                    getName() + " shutdown interrupted",
                    e);
                throw new IOException(e);
            }
        }
    }
}

