package ru.yandex.http.server.sync;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.function.BooleanSupplier;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.HttpResponseInterceptor;
import org.apache.http.HttpServerConnection;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.DefaultHttpResponseFactory;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpService;
import org.apache.http.protocol.ImmutableHttpProcessor;
import org.apache.http.protocol.ResponseServer;

import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.http.util.HeadfulResponseContent;
import ru.yandex.http.util.ResponseConnControl;
import ru.yandex.http.util.ResponseXRequestId;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.StaticHeaders;
import ru.yandex.http.util.YandexReasonPhraseCatalog;
import ru.yandex.http.util.server.BaseServerDynamicConfig;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.http.util.server.ResponseHostname;
import ru.yandex.http.util.server.ServerConfigProvider;
import ru.yandex.http.util.server.SessionContext;
import ru.yandex.http.util.server.SessionId;

public class BaseHttpService extends HttpService {
    // TODO: implement better solution for exception logging
    private final Logger defaultLogger;
    private final ThreadLocal<Logger> logger;

    // CSOFF: ParameterNumber
    public BaseHttpService(
        final CompositeHttpRequestHandlerMapper handlerMapper,
        final Tvm2ServiceContextRenewalTask serviceContextRenewalTask,
        final ServerConfigProvider<ImmutableBaseServerConfig, BaseServerDynamicConfig> config,
        final BooleanSupplier keepAlive)
    {
        super(
            new ImmutableHttpProcessor(
                new HttpRequestInterceptor[] {
                    new SessionContext(
                        handlerMapper,
                        serviceContextRenewalTask,
                        config),
                    new RequestContentEncoding()
                },
                createResponseInterceptors(config.staticConfig(), keepAlive)),
            new DefaultConnectionReuseStrategy(),
            new DefaultHttpResponseFactory(YandexReasonPhraseCatalog.INSTANCE),
            handlerMapper);
        defaultLogger = config.staticConfig().loggers().preparedLoggers().asterisk();
        logger = ThreadLocal.withInitial(() -> defaultLogger);
    }
    // CSON: ParameterNumber

    private static HttpResponseInterceptor[] createResponseInterceptors(
        final ImmutableBaseServerConfig config,
        final BooleanSupplier keepAlive)
    {
        List<HttpResponseInterceptor> responseInterceptors = new ArrayList<>();
        List<Header> staticHeaders = config.staticHeaders();
        if (!staticHeaders.isEmpty()) {
            responseInterceptors.add(
                new StaticHeaders(
                    staticHeaders.toArray(new Header[staticHeaders.size()])));
        }
        responseInterceptors.add(
            new ResponseConnControl(
                keepAlive,
                config.timeout(),
                config.keepAliveStatuses()));
        responseInterceptors.add(
            new ResponseContentEncoding(config.gzip(), config.bufferSize()));
        responseInterceptors.add(HeadfulResponseContent.INSTANCE);
        responseInterceptors.add(ResponseXRequestId.INSTANCE);
        if (!config.briefHeaders()) {
            responseInterceptors.add(SessionId.INSTANCE);
            responseInterceptors.add(ResponseHostname.INSTANCE);
            responseInterceptors.add(new ResponseServer(config.origin()));
        }
        return responseInterceptors.toArray(
            new HttpResponseInterceptor[responseInterceptors.size()]);
    }

    @Override
    public void handleRequest(
        final HttpServerConnection conn,
        final HttpContext context)
        throws HttpException, IOException
    {
        try {
            super.handleRequest(conn, context);
        } finally {
            logger.set(defaultLogger);
        }
    }

    @Override
    protected void handleException(
        final HttpException ex,
        final HttpResponse response)
    {
        logger.get().log(Level.WARNING, "Exception occured", ex);
        response.setHeader(HTTP.CONN_DIRECTIVE, HTTP.CONN_CLOSE);
        if (ex instanceof ServerException) {
            ServerException e = (ServerException) ex;
            response.setStatusCode(e.statusCode());
            response.setEntity(
                new StringEntity(
                    e.toStackTrace(),
                    ContentType.TEXT_PLAIN
                        .withCharset(StandardCharsets.UTF_8)));
        } else {
            super.handleException(ex, response);
        }
    }

    @Override
    protected void doService(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
        Logger logger = (Logger) context.getAttribute(BaseHttpServer.LOGGER);
        this.logger.set(logger);
        super.doService(request, response, context);
        if (logger.isLoggable(Level.FINER)) {
            logger.finer("Response submitted");
        }
    }
}

