package ru.yandex.http.server.sync;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.Socket;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.function.Consumer;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.config.ConnectionConfig;
import org.apache.http.impl.ConnSupport;
import org.apache.http.impl.DefaultBHttpServerConnection;
import org.apache.http.impl.entity.StrictContentLengthStrategy;
import org.apache.http.io.BufferInfo;
import org.apache.http.io.SessionInputBuffer;
import org.apache.http.io.SessionOutputBuffer;
import org.apache.http.protocol.HttpContext;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.server.sync.util.ThreadSafeSocketAdaptor;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.server.HttpSessionInfo;
import ru.yandex.http.util.server.LoggingServerConnection;
import ru.yandex.io.CountingInputStream;
import ru.yandex.io.CountingOutputStream;
import ru.yandex.io.TaggedIOException;
import ru.yandex.io.TaggedInputStream;
import ru.yandex.io.TaggedOutputStream;
import ru.yandex.stater.RequestInfo;
import ru.yandex.util.timesource.TimeSource;

public class LoggingHttpServerConnection
    extends DefaultBHttpServerConnection
    implements LoggingServerConnection
{
    public static final LoggingHttpServerConnection DEAD =
        new LoggingHttpServerConnection();

    protected final ThreadSafeSocketAdaptor socket;
    private final Serializable inputTag = TaggedIOException.createTag();
    private final Serializable outputTag = TaggedIOException.createTag();
    private final SocketChannel channel;
    private final GenericAutoCloseable<RuntimeException> connectionReleaser;
    private final long failFastCheckTimeout;
    private volatile long start = TimeSource.INSTANCE.currentTimeMillis();
    private volatile long processingStart = start;
    private volatile boolean firstRequest = true;
    private volatile boolean inAir = false;
    private InetAddress remoteAddress = null;
    private SelectionKey selectionKey = null;
    private long idleSince = 0L;
    private Logger logger = null;
    private Consumer<RequestInfo> stater = null;
    private volatile GenericAutoCloseable<RuntimeException> requestReleaser =
        null;
    private HttpSessionInfo sessionInfo = null;
    private long bytesRead = 0L;
    private long bytesWritten = 0L;

    private LoggingHttpServerConnection() {
        super(
            1,
            1,
            null,
            null,
            null,
            null,
            null,
            null,
            null);
        channel = null;
        connectionReleaser = null;
        failFastCheckTimeout = 0L;
        socket = null;
    }

    public LoggingHttpServerConnection(
        final BaseHttpServer<?> server,
        final SocketChannel channel,
        final ConnectionConfig config,
        final GenericAutoCloseable<RuntimeException> connectionReleaser)
        throws IOException
    {
        super(
            config.getBufferSize(),
            config.getFragmentSizeHint(),
            ConnSupport.createDecoder(config),
            ConnSupport.createEncoder(config),
            config.getMessageConstraints(),
            StrictContentLengthStrategy.INSTANCE,
            StrictContentLengthStrategy.INSTANCE,
            null,
            null);
        this.channel = channel;
        this.connectionReleaser = connectionReleaser;
        failFastCheckTimeout = server.config().connectionCloseCheckInterval();
        socket = new ThreadSafeSocketAdaptor(server, this);
        bind(socket);
    }

    public long failFastCheckTimeout() {
        return failFastCheckTimeout;
    }

    @Override
    public void bind(final Socket socket) throws IOException {
        super.bind(socket);
        remoteAddress = socket.getInetAddress();
    }

    @Override
    public InetAddress getRemoteAddress() {
        return remoteAddress;
    }

    // BaseHttpServer requires this method to be public, not protected
    @Override
    public Socket getSocket() {
        return super.getSocket();
    }

    //CSOFF: ParameterNumber
    @Override
    public void setRequestContext(
        final Logger logger,
        final Consumer<RequestInfo> stater,
        final GenericAutoCloseable<RuntimeException> requestReleaser,
        final HttpContext context)
    {
        this.logger = logger;
        this.stater = stater;
        this.requestReleaser = requestReleaser;
        sessionInfo = new HttpSessionInfo(context);
        processingStart = TimeSource.INSTANCE.currentTimeMillis();
    }
    //CSON: ParameterNumber

    @Override
    public void setStater(final Consumer<RequestInfo> stater) {
        this.stater = stater;
    }

    @Override
    public void setSessionInfo(final String name, final Object value) {
        sessionInfo.put(name, value);
    }

    public SocketChannel channel() {
        return channel;
    }

    public SelectionKey selectionKey() {
        return selectionKey;
    }

    public long idleSince() {
        return idleSince;
    }

    public void idlefy(final Selector selector) throws IOException {
        idleSince = TimeSource.INSTANCE.currentTimeMillis();
        channel.configureBlocking(false);
        selectionKey = channel.register(selector, SelectionKey.OP_READ, this);
    }

    public void busyfy() throws IOException {
        idleSince = 0L;
        channel.configureBlocking(true);
    }

    @Override
    protected InputStream createInputStream(
        final long len,
        final SessionInputBuffer inbuffer)
    {
        InputStream input = super.createInputStream(len, inbuffer);
        input = new CountingInputStream(input) {
            @Override
            public void close() throws IOException {
                try {
                    super.close();
                } finally {
                    bytesRead = pos();
                }
            }
        };

        return new TaggedInputStream(input, inputTag);
    }

    @Override
    protected OutputStream createOutputStream(
        final long len,
        final SessionOutputBuffer outbuffer)
    {
        OutputStream output = super.createOutputStream(len, outbuffer);
        output = new CountingOutputStream(output) {
            @Override
            public void flush() throws IOException {
                // CountingOutputStream doesn't perform any caching itself, so
                // it is absolutely legal to check pos() before flush()
                bytesWritten = pos();
                super.flush();
            }

            @Override
            public void close() throws IOException {
                bytesWritten = pos();
                super.close();
            }
        };

        return new TaggedOutputStream(output, outputTag);
    }

    public void requestStarted() {
        if (inAir) {
            inAir = false;
        } else {
            if (firstRequest) {
                firstRequest = false;
            } else {
                start = TimeSource.INSTANCE.currentTimeMillis();
            }
        }
    }

    public boolean inAir() {
        return inAir;
    }

    public void setInAir() {
        inAir = true;
    }

    @Override
    public long requestStartTime() {
        return start;
    }

    @Override
    public HttpRequest receiveRequestHeader()
        throws HttpException, IOException
    {
        HttpRequest request = super.receiveRequestHeader();
        bytesRead = 0L;
        bytesWritten = 0L;
        return request;
    }

    private void releaseResources() {
        GenericAutoCloseable<RuntimeException> requestReleaser =
            this.requestReleaser;
        if (requestReleaser != null) {
            requestReleaser.close();
        }
    }

    private void log(final HttpResponse response) {
        log(response, response.getStatusLine().getStatusCode());
    }

    private void log(final HttpResponse response, final int statusCode) {
        releaseResources();
        sessionInfo.put(STATUS, Integer.toString(statusCode));
        sessionInfo.put(REQUEST_LENGTH, Long.toString(bytesRead));
        if (response.getEntity() != null) {
            sessionInfo.put(RESPONSE_LENGTH, Long.toString(bytesWritten));
        }
        RequestInfo info = new RequestInfo(
            TimeSource.INSTANCE.currentTimeMillis(),
            statusCode,
            start,
            processingStart,
            bytesRead,
            bytesWritten);
        sessionInfo.put(REQUEST_TIME, Long.toString(info.requestTime()));
        sessionInfo.put(PROCESSING_TIME, Long.toString(info.processingTime()));
        logger.log(Level.INFO, "", sessionInfo);
        Consumer<RequestInfo> stater = this.stater;
        if (stater != null) {
            stater.accept(info);
        }
    }

    @Override
    public void sendResponseHeader(final HttpResponse response)
        throws HttpException, IOException
    {
        super.sendResponseHeader(response);
        if (response.getStatusLine().getStatusCode() >= HttpStatus.SC_OK) {
            doSendResponseEntity(response);
        }
    }

    private void doSendResponseEntity(final HttpResponse response)
        throws HttpException, IOException
    {
        boolean flush = false;
        long now = TimeSource.INSTANCE.currentTimeMillis();
        try {
            if (now - start > failFastCheckTimeout) {
                flush = true;
                flush(); // fail fast on connections already closed by client
                flush = false;
            }
            super.sendResponseEntity(response);
        } catch (IOException e) {
            IOException ex = e;
            if (flush) {
                log(response, YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
            } else if (e instanceof TaggedIOException
                && outputTag.equals(((TaggedIOException) e).tag()))
            {
                log(response, YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST);
                ex = ((TaggedIOException) e).getCause();
            } else {
                log(response, YandexHttpStatus.SC_REMOTE_CLOSED_REQUEST);
            }
            throw ex;
        } catch (Throwable t) {
            log(response, YandexHttpStatus.SC_ENTITY_SEND_FAILED);
            throw t;
        }
        log(response);
    }

    @Override
    public void sendResponseEntity(final HttpResponse response) {
    }

    public boolean hasBufferedData() {
        SessionInputBuffer inbuf = getSessionInputBuffer();
        if (inbuf instanceof BufferInfo) {
            return ((BufferInfo) inbuf).length() > 0;
        }
        return false;
    }

    public boolean needReregister() throws IOException {
        return false;
    }

    public Serializable inputTag() {
        return inputTag;
    }

    public Serializable outputTag() {
        return outputTag;
    }

    private void releaseConnection() {
        if (connectionReleaser != null) {
            connectionReleaser.close();
        }
    }

    public void socketClose() throws IOException {
        try {
            channel.socket().close();
        } finally {
            releaseResources();
            releaseConnection();
        }
    }

    @Override
    public void close() throws IOException {
        try {
            super.close();
        } finally {
            releaseResources();
            releaseConnection();
        }
    }

    @Override
    public void shutdown() throws IOException {
        try {
            try {
                channel.socket().setSoLinger(true, 0);
            } catch (IOException e) {
            }
            super.shutdown();
        } finally {
            releaseResources();
            releaseConnection();
        }
    }
}
