package ru.yandex.http.server.sync;

import java.io.IOException;
import java.io.InputStream;
import java.util.Locale;
import java.util.zip.GZIPInputStream;

import org.apache.http.Header;
import org.apache.http.HeaderElement;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpRequestInterceptor;
import org.apache.http.client.entity.DecompressingEntity;
import org.apache.http.client.entity.DeflateInputStream;
import org.apache.http.client.entity.InputStreamFactory;
import org.apache.http.config.Lookup;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.UnsupportedMediaTypeException;

public class RequestContentEncoding implements HttpRequestInterceptor {
    private static final InputStreamFactory DEFLATE =
        new InputStreamFactory() {
            @Override
            public InputStream create(final InputStream in)
                throws IOException
            {
                return new DeflateInputStream(in);
            }
        };

    private static final InputStreamFactory GZIP =
        new InputStreamFactory() {
            @Override
            public InputStream create(final InputStream in)
                throws IOException
            {
                return new GZIPInputStream(in);
            }
        };

    private final Lookup<InputStreamFactory> decoders;

    public RequestContentEncoding() {
        this(RegistryBuilder.<InputStreamFactory>create()
            .register("deflate", DEFLATE)
            .register("gzip", GZIP)
            .register("x-gzip", GZIP)
            .build());
    }

    public RequestContentEncoding(final Lookup<InputStreamFactory> decoders) {
        this.decoders = decoders;
    }

    @Override
    public void process(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntityEnclosingRequest entityEnclosingRequest =
                (HttpEntityEnclosingRequest) request;
            HttpEntity entity = entityEnclosingRequest.getEntity();
            if (entity != null && entity.getContentLength() != 0) {
                Header encoding = entity.getContentEncoding();
                if (encoding != null) {
                    for (HeaderElement codec: encoding.getElements()) {
                        String name = codec.getName().toLowerCase(Locale.ROOT);
                        InputStreamFactory decoderFactory =
                            decoders.lookup(name);
                        if (decoderFactory == null) {
                            if (!name.equals("identity")) {
                                throw new UnsupportedMediaTypeException(
                                    "Unsupported Content-Encoding: " + name);
                            }
                        } else {
                            entityEnclosingRequest.setEntity(
                                new DecompressingEntity(
                                    entityEnclosingRequest.getEntity(),
                                    decoderFactory));
                            request.removeHeaders("Content-Length");
                            request.removeHeaders("Content-Encoding");
                            request.removeHeaders("Content-MD5");
                        }
                    }
                }
            }
        }
    }
}

