package ru.yandex.http.util.server;

import ru.yandex.client.tvm2.Tvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ServiceConfigBuilder;
import ru.yandex.http.config.DnsConfig;
import ru.yandex.http.config.DnsConfigBuilder;
import ru.yandex.logger.LoggersConfig;
import ru.yandex.logger.LoggersConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.GolovanPanelConfig;
import ru.yandex.stater.GolovanPanelConfigBuilder;
import ru.yandex.stater.StatersConfig;
import ru.yandex.stater.StatersConfigBuilder;

public abstract class AbstractBaseServerConfigBuilder
    <T extends AbstractBaseServerConfigBuilder<T>>
    extends AbstractHttpServerConfigBuilder<T>
    implements BaseServerConfig
{
    private DnsConfigBuilder dnsConfig;
    private Tvm2ServiceConfigBuilder tvm2ServiceConfig;
    private LoggersConfigBuilder loggers;
    private StatersConfigBuilder staters;
    private AuthsConfigBuilder auths;
    private LimitersConfigBuilder limiters;
    private GolovanPanelConfigBuilder golovanPanel;
    private AutoRegisterRequestStaterConfigBuilder autoRegisterRequestStater;

    protected AbstractBaseServerConfigBuilder(final BaseServerConfig config) {
        super(config);
        dnsConfig(config.dnsConfig());
        tvm2ServiceConfig(config.tvm2ServiceConfig());
        loggers(config.loggers());
        staters(config.staters());
        auths(config.auths());
        limiters(config.limiters());
        golovanPanel(config.golovanPanel());
        autoRegisterRequestStater(config.autoRegisterRequestStater());
    }

    protected AbstractBaseServerConfigBuilder(
        final IniConfig config,
        final BaseServerConfig defaults)
        throws ConfigException
    {
        this(config, DEFAULT_SERVER_SECTION, defaults);
    }

    protected AbstractBaseServerConfigBuilder(
        final IniConfig config,
        final String section,
        final BaseServerConfig defaults)
        throws ConfigException
    {
        super(config.section(section), defaults);
        dnsConfig = new DnsConfigBuilder(
            config.section("dns"),
            defaults.dnsConfig());
        IniConfig tvm2ServiceSection = config.sectionOrNull("tvm2");
        if (tvm2ServiceSection == null) {
            tvm2ServiceConfig(defaults.tvm2ServiceConfig());
        } else {
            Tvm2ServiceConfig tvm2ServiceConfigDefaults =
                defaults.tvm2ServiceConfig();
            if (tvm2ServiceConfigDefaults == null) {
                tvm2ServiceConfig =
                    new Tvm2ServiceConfigBuilder(tvm2ServiceSection);
            } else {
                tvm2ServiceConfig = new Tvm2ServiceConfigBuilder(
                    tvm2ServiceSection,
                    tvm2ServiceConfigDefaults);
            }
        }
        loggers = new LoggersConfigBuilder(config, defaults.loggers());
        staters = new StatersConfigBuilder(config, defaults.staters());
        auths = new AuthsConfigBuilder(config, defaults.auths());
        limiters = new LimitersConfigBuilder(config, defaults.limiters());
        GolovanPanelConfig golovanPanelDefaults = defaults.golovanPanel();
        IniConfig golovanPanelSection = config.sectionOrNull("golovan-panel");
        if (golovanPanelSection == null) {
            if (golovanPanelDefaults == null) {
                golovanPanel = null;
            } else {
                golovanPanel =
                    new GolovanPanelConfigBuilder(golovanPanelDefaults);
            }
        } else {
            if (golovanPanelDefaults == null) {
                golovanPanel =
                    new GolovanPanelConfigBuilder(golovanPanelSection);
            } else {
                golovanPanel = new GolovanPanelConfigBuilder(
                    golovanPanelSection,
                    defaults.golovanPanel());
            }
        }

        IniConfig autoRegisterRequestStaterSection = config.sectionOrNull("auto-request-stater");
        if (autoRegisterRequestStaterSection != null) {
            autoRegisterRequestStater =
                new AutoRegisterRequestStaterConfigBuilder(
                    autoRegisterRequestStaterSection,
                    defaults.autoRegisterRequestStater());
        } else {
            autoRegisterRequestStater =
                new AutoRegisterRequestStaterConfigBuilder(
                    defaults.autoRegisterRequestStater());
        }
    }

    @Override
    public DnsConfigBuilder dnsConfig() {
        return dnsConfig;
    }

    public T dnsConfig(final DnsConfig dnsConfig) {
        this.dnsConfig = new DnsConfigBuilder(dnsConfig);
        return self();
    }

    @Override
    public Tvm2ServiceConfigBuilder tvm2ServiceConfig() {
        return tvm2ServiceConfig;
    }

    public T tvm2ServiceConfig(final Tvm2ServiceConfig tvm2ServiceConfig) {
        if (tvm2ServiceConfig == null) {
            this.tvm2ServiceConfig = null;
        } else {
            this.tvm2ServiceConfig =
                new Tvm2ServiceConfigBuilder(tvm2ServiceConfig);
        }
        return self();
    }

    @Override
    public LoggersConfigBuilder loggers() {
        return loggers;
    }

    public T loggers(final LoggersConfig loggers) {
        this.loggers = new LoggersConfigBuilder(loggers);
        return self();
    }

    @Override
    public StatersConfigBuilder staters() {
        return staters;
    }

    public T staters(final StatersConfig staters) {
        this.staters = new StatersConfigBuilder(staters);
        return self();
    }

    @Override
    public AuthsConfigBuilder auths() {
        return auths;
    }

    public T auths(final AuthsConfig auths) {
        this.auths = new AuthsConfigBuilder(auths);
        return self();
    }

    @Override
    public LimitersConfigBuilder limiters() {
        return limiters;
    }

    public T limiters(final LimitersConfig limiters) {
        this.limiters = new LimitersConfigBuilder(limiters);
        return self();
    }

    @Override
    public GolovanPanelConfigBuilder golovanPanel() {
        return golovanPanel;
    }

    public T golovanPanel(final GolovanPanelConfig golovanPanel) {
        if (golovanPanel == null) {
            this.golovanPanel = null;
        } else {
            this.golovanPanel = new GolovanPanelConfigBuilder(golovanPanel);
        }
        return self();
    }

    @Override
    public AutoRegisterRequestStaterConfigBuilder autoRegisterRequestStater() {
        return autoRegisterRequestStater;
    }

    public AbstractBaseServerConfigBuilder<T> autoRegisterRequestStater(
        final AutoRegisterRequestStaterConfig config)
    {
        this.autoRegisterRequestStater = new AutoRegisterRequestStaterConfigBuilder(config);
        return this;
    }

    public void copyTo(final AbstractBaseServerConfigBuilder<?> other) {
        copyTo((AbstractHttpServerConfigBuilder<?>) other);
        other.dnsConfig(dnsConfig());
        other.tvm2ServiceConfig(tvm2ServiceConfig());
        other.loggers(loggers());
        other.staters(staters());
        other.auths(auths());
        other.limiters(limiters());
        other.golovanPanel(golovanPanel());
        other.autoRegisterRequestStater(autoRegisterRequestStater());
    }
}

