package ru.yandex.http.util.server;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.PrintStream;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.net.InetSocketAddress;
import java.net.URISyntaxException;
import java.nio.channels.SocketChannel;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileStore;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.net.ssl.SSLContext;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.protocol.HttpContext;

import ru.yandex.charset.Encoder;
import ru.yandex.client.tvm2.ImmutableTvm2ServiceConfig;
import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.collection.IdentityHashSet;
import ru.yandex.collection.Iterators;
import ru.yandex.collection.LongPair;
import ru.yandex.collection.Pattern;
import ru.yandex.collection.PatternMap;
import ru.yandex.concurrent.SingleNamedThreadFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.concurrent.TimerCloser;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericNonThrowingCloseableAdapter;
import ru.yandex.function.OutputStreamProcessorAdapter;
import ru.yandex.function.SimpleGenericAutoCloseableHolder;
import ru.yandex.function.TruePredicate;
import ru.yandex.http.config.ImmutableExternalDataConfig;
import ru.yandex.http.config.ImmutableServerHttpsConfig;
import ru.yandex.http.config.ImmutableURICheckConfig;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotFoundException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.DevNullLogger;
import ru.yandex.logger.ImmutableLoggersConfig;
import ru.yandex.logger.LoggerOutputStream;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.logger.RotatableHandler;
import ru.yandex.olelole.config.ImmutableNotificationConfig;
import ru.yandex.olelole.config.NotificationConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.uri.QueryParameter;
import ru.yandex.parser.uri.QueryParser;
import ru.yandex.parser.uri.UriParser;
import ru.yandex.stater.AlertThresholds;
import ru.yandex.stater.AliasingStater;
import ru.yandex.stater.GolovanAlertsConfig;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanPanelConfig;
import ru.yandex.stater.GolovanPanelConfigBuilder;
import ru.yandex.stater.GolovanPanelConfigDefaults;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanAlertsConfig;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.ImmutableStaterConfig;
import ru.yandex.stater.PrefixingStater;
import ru.yandex.stater.RequestsStater;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StaterConfig;
import ru.yandex.stater.StaterConfigBuilder;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.system.CPUMonitor;
import ru.yandex.util.system.RusageMonitor;
import ru.yandex.util.timesource.TimeSource;

public abstract class AbstractHttpServer
    <C extends ImmutableBaseServerConfig, T>
    implements HttpServer<C, T>,
        Runnable,
        ServerConfigProvider<ImmutableBaseServerConfig, BaseServerDynamicConfig>
{
    static {
        Thread.setDefaultUncaughtExceptionHandler(
            UncaughtExceptionHandler.INSTANCE);
    }

    private static final long PERCENTS = 100L;
    private static final int MILLIS_PER_SECOND = 1000;
    private static final String LOG_ROTATE_FAILED = "Log rotate failed";
    private static final Object OBJECT = new Object();
    private static final Logger[] EMPTY_LOGGERS = new Logger[0];
    private static final Stater[] EMPTY_STATERS = new Stater[0];
    private static final ExternalDataUpdater[] EMPTY_UPDATERS =
        new ExternalDataUpdater[0];

    protected GenericAutoCloseableChain<IOException> closeChain =
        new GenericAutoCloseableChain<>();
    protected final C config;
    protected final BaseServerDynamicConfig dynamicConfig;
    protected final RequestHandlerMapper<T> handlerMapper;
    protected final Thread thread;
    protected final PrefixedLogger logger;
    protected final ImmutableGolovanPanelConfig golovanPanelConfig;
    protected final TimeFrameQueue<Object> incomingConnections;
    protected final TimeFrameQueue<Object> limitedConnections;
    protected final Tvm2ServiceContextRenewalTask serviceContextRenewalTask;
    protected final ImmutableServerHttpsConfig httpsConfig;
    protected final SSLContext sslContext;
    protected final RusageMonitor rusageMon;
    protected final IBMDumper ibmDumper;
    protected volatile Set<String> debugFlags = Collections.emptySet();

    private final PatternMap<RequestInfo, RequestsStater> requestsStaters;
    private final Timer httpCheckersTimer;
    private final Timer externalDataUpdateTimer;
    private final HttpChecker[] httpCheckers;
    private volatile Logger[] loggers = EMPTY_LOGGERS;
    private volatile Stater[] staters = EMPTY_STATERS;
    private volatile ExternalDataUpdater[] externalDataUpdaters =
        EMPTY_UPDATERS;
    private volatile boolean pingEnabled;

    protected AbstractHttpServer(
        final C config,
        final RequestHandlerMapper<T> handlerMapper)
        throws IOException
    {
        this.config = config;
        this.requestsStaters = config.staters().preparedStaters();
        this.handlerMapper = handlerMapper;
        pingEnabled = config.pingEnabledOnStartup();
        for (String debugFlag: config.debugFlags()) {
            addDebugFlag(debugFlag);
        }
        thread =
            new Thread(new ThreadGroup(config.name()), this, config.name());
        logger = config.loggers().preparedLoggers().asterisk();
        this.dynamicConfig = new BaseServerDynamicConfig(config);
        golovanPanelConfig = config.golovanPanel();
        incomingConnections = new TimeFrameQueue<>(config.metricsTimeFrame());
        limitedConnections = new TimeFrameQueue<>(config.metricsTimeFrame());
        ImmutableTvm2ServiceConfig tvm2ServiceConfig =
            config.tvm2ServiceConfig();
        if (tvm2ServiceConfig == null) {
            serviceContextRenewalTask = null;
        } else {
            try {
                serviceContextRenewalTask = new Tvm2ServiceContextRenewalTask(
                    logger.addPrefix("TVM2"),
                    tvm2ServiceConfig,
                    config.dnsConfig());
                closeChain.add(serviceContextRenewalTask);
            } catch (HttpException | JsonException | URISyntaxException e) {
                throw new IOException(
                    "Failed to initialized TVM 2.0 service context",
                    e);
            }
        }

        httpsConfig = config.httpsConfig();
        long certificateValidUntil;
        if (httpsConfig == null) {
            sslContext = null;
            certificateValidUntil = Long.MAX_VALUE;
        } else {
            sslContext = httpsConfig.getSSLContext();
            certificateValidUntil = httpsConfig.certificateValidUntil();
        }

        Map<String, ImmutableURICheckConfig> httpChecks = config.httpChecks();
        if (httpChecks.isEmpty()) {
            httpCheckersTimer = null;
            httpCheckers = new HttpChecker[0];
        } else {
            httpCheckersTimer = new Timer("HttpChecker", true);
            closeChain.add(new TimerCloser(httpCheckersTimer));
            int size = httpChecks.size();
            List<HttpChecker> httpCheckers = new ArrayList<>(size);
            for (Map.Entry<String, ImmutableURICheckConfig> entry
                : httpChecks.entrySet())
            {
                String name = entry.getKey();
                ImmutableURICheckConfig checkConfig = entry.getValue();
                try {
                    HttpChecker checker =
                        new HttpChecker(
                            name,
                            logger.addPrefix(name),
                            checkConfig,
                            config.dnsConfig());
                    closeChain.add(checker);
                    httpCheckersTimer.schedule(
                        checker,
                        checkConfig.checkInterval(),
                        checkConfig.checkInterval());
                    httpCheckers.add(checker);
                } catch (HttpException | IOException e) {
                    throw new IOException("HTTP check failed: " + name, e);
                }
            }
            this.httpCheckers = httpCheckers.toArray(new HttpChecker[size]);
        }

        externalDataUpdateTimer = new Timer("ExternalData", true);
        closeChain.add(new TimerCloser(externalDataUpdateTimer));
        Path localCacheDir = config.localCacheDir();
        if (localCacheDir != null) {
            Files.createDirectories(config.localCacheDir());
        }
        for (Map.Entry<String, ImmutableExternalDataConfig> entry
            : config.externalData().entrySet())
        {
            try {
                addExternalDataUpdater(entry.getKey(), entry.getValue());
            } catch (ConfigException | HttpException e) {
                throw new IOException(e);
            }
        }

        registerStater(new HealthStater(certificateValidUntil));

        ImmutableLoggersConfig loggers = config.loggers();
        Logger stdout = loggers.loggerForStdout();
        if (stdout != null) {
            registerLoggerForLogrotate(stdout);
        }
        Logger stderr = loggers.loggerForStderr();
        if (stderr != null) {
            registerLoggerForLogrotate(stderr);
        }
        loggers.preparedLoggers().traverse(
            (pattern, logger) -> registerLoggerForLogrotate(logger));
        loggers.preparedAccessLoggers().traverse(
            (pattern, logger) -> registerLoggerForLogrotate(logger));

        registerStaters(config.staters());
        config.limiters().preparedLimiters().traverse(
            (pattern, limiter) -> {
                if (limiter.staterEnabled()) {
                    registerStater(limiter);
                }
            });

        PrefixedLogger memGcLogger = logger;
        if (!Boolean.parseBoolean(System.getProperty(
            "VERBOSE_MEM_GC_LOG",
            Boolean.TRUE.toString())))
        {
            memGcLogger = DevNullLogger.INSTANCE;
        }

        CPUMonitor cpuMon = null;
        RusageMonitor rusageMon = null;
        MemoryMonitor memoryMon = null;
        GCMonitor gcMon = null;
        try {
            if (config.cpuStater()) {
                cpuMon = new CPUMonitor();
                rusageMon = new RusageMonitor(
                    config.metricsTimeFrame(),
                    config.timerResolution(),
                    e -> logger.log(Level.WARNING, "Failed to get rusage", e),
                    new SingleNamedThreadFactory(
                        getThreadGroup(),
                        config.name() + "-Rusage",
                        true));
                closeChain.add(
                    new GenericNonThrowingCloseableAdapter<>(rusageMon));
            }
            if (config.memoryStater()) {
                memoryMon = MemoryMonitor.instance(memGcLogger);
            }
            if (config.gcStater()) {
                gcMon = GCMonitor.instance(memGcLogger);
            }
        } catch (ReflectiveOperationException e) {
            cpuMon = null;
            rusageMon = null;
            memoryMon = null;
            gcMon = null;
            logger.log(
                Level.INFO,
                "Failed to construct CPU/Memory Monitor, cpu/memory"
                    + " usage won't be available",
                e);
        }
        this.rusageMon = rusageMon;
        registerStater(
            new SystemInfoStater(
                cpuMon,
                rusageMon,
                memoryMon,
                gcMon));

        IBMDumper ibmDumper;
        try {
            ibmDumper = new IBMDumper();
        } catch (ReflectiveOperationException e) {
            ibmDumper = null;
            logger.info(
                "Failed to construct IBMDumper, dumps won't be available");
        }
        this.ibmDumper = ibmDumper;

        List<LongPair<String>> staticStats = config.staticStats();
        if (!staticStats.isEmpty()) {
            registerStater(new StaticStatsStater(staticStats));
        }

        if (config.instanceAliveStater()) {
            registerStater(new InstanceAliveStater());
        }

        if (config.heapStater()) {
            registerStater(HeapStater.INSTANCE);
        }

        registerStater(
            new IncomingConnectionsStater(
                incomingConnections,
                limitedConnections,
                ((double) MILLIS_PER_SECOND) / config.metricsTimeFrame()));
        registerStater(
            new ConnectionsLimiterStater(config.connectionsLimiter()));

        Map<String, FileStore> freeSpaceSignals = config.freeSpaceSignals();
        if (!freeSpaceSignals.isEmpty()) {
            registerStater(new FreeSpaceStater(freeSpaceSignals, logger));
        }

        if (golovanPanelConfig != null) {
            List<String> diskVolumes = golovanPanelConfig.diskVolumes();
            if (!diskVolumes.isEmpty()) {
                registerStater(new DiskVolumesStater(diskVolumes));
            }
        }

        Map<String, ImmutableFilesStaterConfig> filesStaters =
            config.filesStaters();
        if (!filesStaters.isEmpty()) {
            registerStater(new FilesStater(filesStaters, logger));
        }
    }

    protected abstract T wrap(T handler);

    protected abstract T unwrap(T handler);

    public String getName() {
        return thread.getName();
    }

    public ThreadGroup getThreadGroup() {
        return thread.getThreadGroup();
    }

    public void interrupt() {
        thread.interrupt();
    }

    @Override
    public void join() throws InterruptedException {
        thread.join();
    }

    @Override
    public SSLContext sslContext() {
        return sslContext;
    }

    @Override
    public boolean hasIBMDumper() {
        return ibmDumper != null;
    }

    @Override
    public boolean pingEnabled(final PrefixedLogger logger) {
        if (pingEnabled) {
            for (HttpChecker httpChecker: httpCheckers) {
                if (!httpChecker.ok()) {
                    if (logger != null) {
                        logger.warning(
                            "/ping disabled because of failed "
                            + httpChecker.name() + " checker");
                    }
                    return false;
                }
            }
            if (serviceContextRenewalTask == null) {
                return true;
            } else {
                Tvm2ServiceContextRenewalTask.Status status =
                    serviceContextRenewalTask.status();
                if (status == Tvm2ServiceContextRenewalTask.Status.EXPIRED) {
                    if (logger != null) {
                        logger.warning(
                            "/ping disabled because of expired TVM keys");
                    }
                    return false;
                } else {
                    return true;
                }
            }
        } else {
            if (logger != null) {
                logger.warning("/ping disabled by flag");
            }
            return false;
        }
    }

    @Override
    public void enablePing() {
        pingEnabled = true;
    }

    @Override
    public void disablePing() {
        pingEnabled = false;
    }

    @Override
    public Set<String> debugFlags() {
        return debugFlags;
    }

    @Override
    public void addDebugFlag(final String debugFlag) {
        String interned = debugFlag.intern();
        if (this.debugFlags.isEmpty()) {
            this.debugFlags = Collections.singleton(interned);
        } else {
            Set<String> debugFlags = new IdentityHashSet<>(this.debugFlags);
            debugFlags.add(interned);
            this.debugFlags = debugFlags;
        }
    }

    @Override
    public void removeDebugFlag(final String debugFlag) {
        Set<String> debugFlags = new IdentityHashSet<>(this.debugFlags);
        debugFlags.remove(debugFlag.intern());
        int size = debugFlags.size();
        switch (size) {
            case 0:
                debugFlags = Collections.emptySet();
                break;
            case 1:
                debugFlags =
                    Collections.singleton(debugFlags.iterator().next());
                break;
            default:
                break;
        }
        this.debugFlags = debugFlags;
    }

    private synchronized void registerStaterForHandler(final Pattern<RequestInfo> pattern) {
        StaterConfig defaultConfig =
            config.autoRegisterRequestStater().staterConfig();
        ImmutableStaterConfig predefinedStater =
            config.staters().staters().get(pattern);
        StaterConfigBuilder staterConfig =
            new StaterConfigBuilder(defaultConfig);
        switch (config.autoRegisterRequestStater().status()) {
            case ALL:
                break;
            case UNCONFIGURED:
                if (predefinedStater != config.staters().staters().asterisk()) {
                    return;
                }

                break;
            default:
                return;
        }

        if (predefinedStater != config.staters().staters().asterisk()) {
            logger.fine(
                "Skipping register autostater for " + pattern
                + " — was predefined in config by " + predefinedStater.build());
            return;
        }

        StringBuilder signalPrefix = new StringBuilder();
        String path = pattern.path();

        path = path
            .replaceAll("/", "_")
            .replaceAll("[^a-z0-9_]+", "_")
            .replaceAll("_+", "_");
        if (path.startsWith("_") && path.length() > 1) {
            path = path.substring(1);
        }

        if (path.endsWith("_") && path.length() > 1) {
            path = path.substring(0, path.length() - 1);
        }

        signalPrefix.append(path);
        if (pattern.prefix()) {
            if (signalPrefix.length() > 0) {
                signalPrefix.append('_');
            }

            signalPrefix.append("asterisk");
        }

        if (signalPrefix.length() == 0) {
            signalPrefix.append("root__route");
        }

        if (!TruePredicate.INSTANCE.equals(pattern.predicate())) {
            signalPrefix.append('_');
            String predicateString = pattern.predicate().toString()
                .toLowerCase(Locale.ENGLISH)
                .replaceAll("[^a-z0-9_]+", "_")
                .replaceAll("_+", "_");
            if (predicateString.startsWith("_")) {
                predicateString = predicateString.substring(1);
            }

            if (predicateString.endsWith("_")) {
                predicateString = predicateString.substring(0, predicateString.length() - 1);
            }

            signalPrefix.append(predicateString);
        }

        // golovan limit https://wiki.yandex-team.ru/golovan/userdocs/tagsandsignalnaming/
        if (signalPrefix.length() >= 128) {
            staterConfig.prefix(signalPrefix.toString().substring(0, 127));
        } else {
            staterConfig.prefix(signalPrefix.toString());
        }

        RequestsStater stater = staterConfig.build().build();
        registerStater(stater);
        requestsStaters.put(pattern, stater);
    }

    public void register(
        final Pattern<RequestInfo> pattern,
        final T handler,
        final boolean addRequestStater)
    {
        T wrapped = wrap(handler);
        for (String method: config.defaultHttpMethods()) {
            handlerMapper.register(pattern, wrapped, method);
        }
        if (addRequestStater) {
            registerStaterForHandler(pattern);
        }
    }

    @Override
    public void register(final Pattern<RequestInfo> pattern, final T handler) {
        register(pattern, handler, true);
    }

    @Override
    public void register(
        final Pattern<RequestInfo> pattern,
        final T handler,
        final String... methods)
    {
        handlerMapper.register(pattern, wrap(handler), methods);
        registerStaterForHandler(pattern);
    }

    @Override
    public T register(
        final Pattern<RequestInfo> pattern,
        final T handler,
        final String method)
    {
        registerStaterForHandler(pattern);
        return unwrap(handlerMapper.register(pattern, wrap(handler), method));
    }

    @Override
    public void unregister(final Pattern<RequestInfo> pattern) {
        handlerMapper.unregister(pattern);
    }

    @Override
    public void unregister(
        final Pattern<RequestInfo> pattern,
        final String... methods)
    {
        handlerMapper.unregister(pattern, methods);
    }

    @Override
    public T unregister(
        final Pattern<RequestInfo> pattern,
        final String method)
    {
        return unwrap(handlerMapper.unregister(pattern, method));
    }

    @Override
    @SuppressWarnings("ReferenceEquality")
    public void start() throws IOException {
        InetSocketAddress address = address();
        InetSocketAddress httpAddress = httpAddress();
        StringBuilder sb = new StringBuilder("Starting ");
        sb.append(config.name());
        sb.append(" on ");
        sb.append(address.toString());
        sb.append('(');
        sb.append(scheme());
        if (sslContext != null) {
            sb.append('[');
            sb.append(sslContext.getProvider().getName());
            sb.append(' ');
            sb.append(sslContext.getProtocol());
            sb.append(']');
        }
        if (httpAddress != address) {
            sb.append(") and ");
            sb.append(httpAddress.toString());
            sb.append("(http");
        }
        sb.append("), workers: ");
        sb.append(config.workers());
        logger.info(new String(sb));
        if (serviceContextRenewalTask != null) {
            serviceContextRenewalTask.start();
        }
        if (rusageMon != null) {
            rusageMon.start();
        }
        thread.start();
    }

    @Override
    public void close() throws IOException {
        logger.info(getName() + " shutdown started");
        closeChain.close();
        logger.info(getName() + " shutdown completed");
    }

    @Override
    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public synchronized void registerLoggerForLogrotate(final Logger logger) {
        for (Logger oldLogger: loggers) {
            if (logger.equals(oldLogger)) {
                // This logger already present
                return;
            }
        }
        int len = loggers.length;
        Logger[] loggers = Arrays.copyOf(this.loggers, len + 1);
        loggers[len] = logger;
        this.loggers = loggers;
    }

    @Override
    public void registerLoggerForLogrotate(final HttpServer<?, ?> other) {
        for (Logger logger: other.loggers()) {
            registerLoggerForLogrotate(logger);
        }
    }

    @Override
    public void registerStater(Stater stater) {
        String statsPrefix = config.statsPrefix();
        stater = AliasingStater.create(stater, config.statsAliases());
        if (statsPrefix != null) {
            stater = new PrefixingStater(
                statsPrefix,
                stater,
                config.keepUnprefixedStats());
        }
        synchronized (this) {
            int len = staters.length;
            Stater[] staters = Arrays.copyOf(this.staters, len + 1);
            staters[len] = stater;
            this.staters = staters;
        }
    }

    @Override
    public final List<Logger> loggers() {
        return Arrays.asList(loggers);
    }

    @Override
    public PatternMap<RequestInfo, RequestsStater> requestsStaters() {
        return requestsStaters;
    }

    // Made final in order to simplify stats aliases and have control over all
    // staters
    @Override
    public final <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        Stater[] staters = this.staters;
        for (Stater stater: staters) {
            stater.stats(statsConsumer);
        }
    }

    @Override
    public final void addToGolovanPanel(
        final GolovanPanel panel,
        final String prefix)
    {
        Stater[] staters = this.staters;
        for (Stater stater: staters) {
            stater.addToGolovanPanel(panel, prefix);
        }
    }

    @Override
    public final void addToAlertsConfig(
        final IniConfig alertsConfig,
        final ImmutableGolovanPanelConfig panelConfig,
        final String prefix)
        throws BadRequestException
    {
        Stater[] staters = this.staters;
        for (Stater stater: staters) {
            stater.addToAlertsConfig(alertsConfig, golovanPanelConfig, prefix);
        }
    }

    @Override
    public void rotateLogs(final HttpContext context) throws HttpException {
        Logger sessionLogger = (Logger) context.getAttribute(LOGGER);
        Set<RotatableHandler> handlers = new HashSet<>();
        Logger[] loggers = this.loggers;
        for (Logger logger: loggers) {
            for (Handler handler: logger.getHandlers()) {
                if (handler instanceof RotatableHandler) {
                    handlers.add((RotatableHandler) handler);
                }
            }
        }
        try {
            for (RotatableHandler handler: handlers) {
                handler.rotate();
            }
        } catch (FileNotFoundException e) {
            sessionLogger.log(Level.WARNING, LOG_ROTATE_FAILED, e);
            throw new ServiceUnavailableException(LOG_ROTATE_FAILED, e);
        }
        sessionLogger.info("Logs rotated");
    }

    @Override
    public C config() {
        return config;
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put(
            ACTIVE_CONNECTIONS,
            config.connectionsLimiter().concurrentRequests());
        return status;
    }

    @Override
    public void onIncomingConnection(final SocketChannel channel) {
        incomingConnections.accept(OBJECT);
        if (logger.isLoggable(Level.INFO)) {
            logger.info("Incoming connection: " + channel);
        }
    }

    @Override
    public void onLimitedConnection(
        final SocketChannel channel,
        final String message)
    {
        limitedConnections.accept(OBJECT);
        if (logger.isLoggable(Level.WARNING)) {
            logger.warning(
                "Connection blocked by limiter: " + channel
                + '.' + ' ' + message);
        }
    }

    @Override
    public void discardChannel(final SocketChannel channel) {
        if (logger.isLoggable(Level.WARNING)) {
            logger.warning("Discarding channel: " + channel);
        }
        try {
            channel.socket().setSoLinger(true, 0);
            channel.shutdownOutput();
        } catch (IOException e) {
        }
        try {
            channel.close();
        } catch (IOException e) {
        }
    }

    private ImmutableGolovanPanelConfig customGolovanPanelConfig(
        final HttpRequest request,
        final GolovanPanelConfig defaults)
        throws BadRequestException
    {
        String uri = request.getRequestLine().getUri();
        QueryParser parser = new UriParser(uri).queryParser();
        try {
            IniConfig ini = IniConfig.empty();
            boolean allDefault = defaults == golovanPanelConfig;
            for (QueryParameter param: parser) {
                ini.put(param.name(), param.value().decode());
                allDefault = false;
            }
            if (allDefault) {
                // Nothing changed, we can return current config
                return golovanPanelConfig;
            }
            GolovanPanelConfigBuilder panelConfig =
                new GolovanPanelConfigBuilder(ini, defaults);

            panelConfig.title(panelConfig.title() + ' ' + uri);
            GolovanAlertsConfig alertsConfig = panelConfig.alerts();
            if (alertsConfig == null) {
                panelConfig.alerts(defaults.alerts());
            }
            return panelConfig.build();
        } catch (ConfigException e) {
            throw new BadRequestException(e);
        }
    }

    @Override
    public GolovanPanel customGolovanPanel(final HttpRequest request)
        throws HttpException
    {
        return customGolovanPanel(
            request,
            GolovanPanelConfigDefaults.INSTANCE);
    }

    private GolovanPanel customGolovanPanel(
        final HttpRequest request,
        final GolovanPanelConfig defaults)
        throws BadRequestException
    {
        ImmutableGolovanPanelConfig config =
            customGolovanPanelConfig(request, defaults);
        GolovanPanel panel = new GolovanPanel(config);
        addToGolovanPanel(panel, config.prefix());
        return panel;
    }

    @Override
    public GolovanPanel golovanPanel(final HttpRequest request)
        throws HttpException
    {
        return customGolovanPanel(request, golovanPanelConfig);
    }

    private IniConfig customAlerts(
        final HttpRequest request,
        final GolovanPanelConfig defaults)
        throws HttpException
    {
        ImmutableGolovanPanelConfig panelConfig =
            customGolovanPanelConfig(request, defaults);
        ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
        if (alerts == null) {
            throw new NotFoundException("No alerts config found");
        }

        IniConfig alertsConfig = IniConfig.empty();
        alertsConfig.put("abc", panelConfig.abc());
        alertsConfig.put("namespace", alerts.namespace());
        alertsConfig.put("module", alerts.module());
        String yasmPanel = alerts.yasmPanel();
        if (yasmPanel != null) {
            alertsConfig.put("yasm-panel", yasmPanel);
        }
        String yasmTemplate = alerts.yasmTemplate();
        if (yasmTemplate != null) {
            alertsConfig.put("yasm-template", yasmTemplate);
        }
        if (panelConfig != golovanPanelConfig) {
            alertsConfig.put(
                "alerts-config-uri",
                request.getRequestLine().getUri());
        }
        String tag = panelConfig.tag();
        for (String part: tag.split(";")) {
            if (!part.isEmpty()) {
                int idx = part.indexOf('=');
                if (idx != -1) {
                    alertsConfig.put(
                        part.substring(0, idx),
                        part.substring(idx + 1));
                }
            }
        }
        for (Map.Entry<String, ImmutableNotificationConfig> entry
            : alerts.notificationsConfigs().entrySet())
        {
            NotificationConfig.toIniConfig(
                alertsConfig.section("notification." + entry.getKey()),
                entry.getValue());
        }

        for (Map.Entry<String, ImmutableNotificationConfig> entry
            : alerts.disasterNotificationsConfigs().entrySet())
        {
            NotificationConfig.toIniConfig(
                alertsConfig.section("notification." + entry.getKey()),
                entry.getValue());
        }

        addToAlertsConfig(alertsConfig, panelConfig, panelConfig.prefix());
        return alertsConfig;
    }

    @Override
    public IniConfig alerts(final HttpRequest request)
        throws HttpException
    {
        return customAlerts(request, golovanPanelConfig);
    }

    @Override
    public IniConfig customAlerts(final HttpRequest request)
        throws HttpException
    {
        return customAlerts(request, GolovanPanelConfigDefaults.INSTANCE);
    }

    private void addExternalDataUpdater(final ExternalDataUpdater updater)
        throws ConfigException
    {
        String name = updater.name();
        synchronized (this) {
            for (ExternalDataUpdater oldUpdater
                : this.externalDataUpdaters)
            {
                if (name.equals(oldUpdater.name())) {
                    throw new ConfigException(
                        "External data source already registered for name "
                        + name);
                }
            }
            int len = this.externalDataUpdaters.length;
            ExternalDataUpdater[] externalDataUpdaters =
                Arrays.copyOf(this.externalDataUpdaters, len + 1);
            externalDataUpdaters[len] = updater;
            this.externalDataUpdaters = externalDataUpdaters;
        }
        externalDataUpdateTimer.schedule(
            updater,
            updater.updateInterval(),
            updater.updateInterval());
    }

    @Override
    public void addExternalDataUpdater(
        final String name,
        final ImmutableExternalDataConfig config)
        throws ConfigException, HttpException, IOException
    {
        ExternalDataUpdater updater =
            new ExternalDataUpdater(
                name,
                this,
                serviceContextRenewalTask,
                config);
        try (SimpleGenericAutoCloseableHolder<IOException> holder =
                new SimpleGenericAutoCloseableHolder<>(updater))
        {
            addExternalDataUpdater(updater);
            closeChain.add(holder.release());
        }
    }

    @Override
    public void addExternalDataUpdater(
        final String name,
        final ImmutableExternalDataConfig config,
        final ExternalDataSubscriber subscriber)
        throws ConfigException, HttpException, IOException
    {
        ExternalDataUpdater updater =
            new ExternalDataUpdater(
                name,
                this,
                serviceContextRenewalTask,
                config);
        try (SimpleGenericAutoCloseableHolder<IOException> holder =
                new SimpleGenericAutoCloseableHolder<>(updater))
        {
            addExternalDataUpdater(updater);
            updater.subscribe(subscriber);
            closeChain.add(holder.release());
        }
    }

    @Override
    public void subscribeForExternalDataUpdates(
        final String name,
        final ExternalDataSubscriber subscriber)
        throws ConfigException, HttpException, IOException
    {
        ExternalDataUpdater[] externalDataUpdaters = this.externalDataUpdaters;
        for (ExternalDataUpdater updater: externalDataUpdaters) {
            if (name.equals(updater.name())) {
                updater.subscribe(subscriber);
                return;
            }
        }
        throw new ConfigException(
            "No external data source found for name '" + name + '\'');
    }

    // For testing purposes
    public void updateExternalData() {
        for (ExternalDataUpdater updater: externalDataUpdaters) {
            updater.run();
        }
    }

    @Override
    public ImmutableBaseServerConfig staticConfig() {
        return config;
    }

    @Override
    public BaseServerDynamicConfig dynamicConfig() {
        return dynamicConfig;
    }

    public void updateDynamicConfig(final IniConfig config)
        throws ConfigException
    {
        updateDynamicConfig(config, logger);
    }

    public synchronized void updateDynamicConfig(
        final IniConfig config,
        final PrefixedLogger logger)
        throws ConfigException
    {
        dynamicConfig.update(config, logger);
    }

    private static long booleanToLong(final boolean value) {
        if (value) {
            return 1L;
        } else {
            return 0L;
        }
    }

    private static class SystemInfoStater implements Stater {
        private final CPUMonitor cpuMonitor;
        private final RusageMonitor rusageMon;
        private final MemoryMonitor memoryMonitor;
        private final GCMonitor gcMonitor;

        SystemInfoStater(
            final CPUMonitor cpuMonitor,
            final RusageMonitor rusageMon,
            final MemoryMonitor memoryMonitor,
            final GCMonitor gcMonitor)
        {
            this.cpuMonitor = cpuMonitor;
            this.rusageMon = rusageMon;
            this.memoryMonitor = memoryMonitor;
            this.gcMonitor = gcMonitor;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            if (cpuMonitor != null) {
                double cpuUsage = cpuMonitor.getCpuUsage();
                long upTime = cpuMonitor.getUpTimeSecs();
                double systemCpuLoad = cpuMonitor.getSystemCpuLoad();
                int cpuCount = cpuMonitor.cpuCount();
                statsConsumer.stat("process-cpu-usage_avvv", cpuUsage);
                statsConsumer.stat("process-cpu-usage_ammm", cpuUsage);
                statsConsumer.stat("process-cpu-usage_axxx", cpuUsage);
                statsConsumer.stat("system-cpu-usage_avvv", systemCpuLoad);
                statsConsumer.stat("system-cpu-usage_ammm", systemCpuLoad);
                statsConsumer.stat("system-cpu-usage_axxx", systemCpuLoad);
                statsConsumer.stat("uptime_avvv", upTime);
                statsConsumer.stat("uptime_ammm", upTime);
                statsConsumer.stat("uptime_axxx", upTime);
                statsConsumer.stat("cpu-count_ammv", cpuCount);
                statsConsumer.stat("cpu-count_ammx", cpuCount);
                statsConsumer.stat("cpu-count_ammn", cpuCount);
                statsConsumer.stat("cpu-count_annn", cpuCount);
                statsConsumer.stat("cpu-count_axxx", cpuCount);
            }
            if (rusageMon != null) {
                RusageMonitor.Snapshot snapshot = rusageMon.snapshot();
                double userUsage = RusageMonitor.userUsage(
                    snapshot.startRecord(),
                    snapshot.endRecord());
                double systemUsage = RusageMonitor.systemUsage(
                    snapshot.startRecord(),
                    snapshot.endRecord());
                statsConsumer.stat("rusage-user_ammm", userUsage);
                statsConsumer.stat("rusage-user_axxx", userUsage);
                statsConsumer.stat("rusage-system_ammm", systemUsage);
                statsConsumer.stat("rusage-system_axxx", systemUsage);
                statsConsumer.stat(
                    "rusage-total_axxx",
                    userUsage + systemUsage);
                statsConsumer.stat(
                    "peak-cpu-usage_axxx",
                    snapshot.peakCPUUsage());
                statsConsumer.stat(
                    "peak-cpu-timestamp_axxx",
                    snapshot.peakEndRecord().timestamp());
            }
            long anonymousMemoryUsage;
            if (memoryMonitor == null) {
                anonymousMemoryUsage = 0L;
            } else {
                anonymousMemoryUsage =
                    memoryMonitor.anonymousMemoryUsage();
                long memoryLimit = memoryMonitor.memoryLimit();
                statsConsumer.stat(
                    "process-anonymous-memory-usage_avvv",
                    anonymousMemoryUsage);
                statsConsumer.stat(
                    "process-anonymous-memory-usage_axxx",
                    anonymousMemoryUsage);
                statsConsumer.stat(
                    "process-anonymous-memory-usage_ammm",
                    anonymousMemoryUsage);

                statsConsumer.stat(
                    "process-memory-limit_avvv",
                    memoryLimit);
                statsConsumer.stat(
                    "process-memory-limit_axxx",
                    memoryLimit);
                statsConsumer.stat(
                    "process-memory-limit_ammm",
                    memoryLimit);

                if (memoryLimit > 0) {
                    double usagePct =
                        ((double) (anonymousMemoryUsage * PERCENTS))
                            / memoryLimit;
                    statsConsumer.stat(
                        "process-anonymous-memory-usage-pct_axxx",
                        usagePct);
                }
            }
            if (gcMonitor != null) {
                final long heapUsage = gcMonitor.heapUsage();
                statsConsumer.stat(
                    "heap-memory-usage_avvv",
                    heapUsage);
                statsConsumer.stat(
                    "heap-memory-usage_axxx",
                    heapUsage);
                statsConsumer.stat(
                    "heap-memory-usage_ammm",
                    heapUsage);
                statsConsumer.stat(
                    "heap-memory-usage_annn",
                    heapUsage);
            }
            if (gcMonitor != null && memoryMonitor != null) {
                long heapSize = gcMonitor.memoryLimit();
                long nativeOverhead =
                    Math.max(anonymousMemoryUsage - heapSize, 0);
                statsConsumer.stat(
                    "native-memory-overhead_avvv",
                    nativeOverhead);
                statsConsumer.stat(
                    "native-memory-overhead_axxx",
                    nativeOverhead);
                statsConsumer.stat(
                    "native-memory-overhead_ammm",
                    nativeOverhead);
                statsConsumer.stat(
                    "native-memory-overhead_annn",
                    nativeOverhead);
            }
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            if (memoryMonitor != null) {
                GolovanChartGroup group =
                    new GolovanChartGroup(statsPrefix, statsPrefix);

                GolovanChart absolutes = new GolovanChart(
                    "anonymous-memory",
                    " anonymous memory usage (bytes)",
                    false,
                    false,
                    0d);
                absolutes.addSignal(
                    new GolovanSignal(
                        statsPrefix + "process-anonymous-memory-usage_axxx",
                        config.tag(),
                        "usage",
                        null,
                        0,
                        false));
                absolutes.addSignal(
                    new GolovanSignal(
                        statsPrefix + "process-memory-limit_axxx",
                        config.tag(),
                        "max",
                        null,
                        0,
                        false));
                group.addChart(absolutes);

                GolovanChart pct = new GolovanChart(
                    "anonymous-memory-pct",
                    " anonymous memory usage (%)",
                    false,
                    false,
                    0d);
                pct.addSignal(
                    new GolovanSignal(
                        "const(100)",
                        config.tag(),
                        "100%",
                        "#ff0000",
                        0,
                        false));
                pct.addSignal(
                    new GolovanSignal(
                        "const(75)",
                        config.tag(),
                        "75%",
                        "#ff8000",
                        0,
                        false));
                pct.addSignal(
                    new GolovanSignal(
                        statsPrefix
                        + "process-anonymous-memory-usage-pct_axxx",
                        config.tag(),
                        "usage",
                        null,
                        0,
                        false));
                group.addChart(pct);

                panel.addCharts(
                    GolovanPanelConfig.CATEGORY_MEMORY,
                    null,
                    group);
            }
            if (rusageMon != null) {
                GolovanChartGroup group =
                    new GolovanChartGroup(statsPrefix, statsPrefix);
                GolovanChart chart = new GolovanChart(
                    "cpu-usage",
                    " Java CPU usage (%)",
                    false,
                    false,
                    0d);
                chart.addSignal(
                    new GolovanSignal(
                        "mul(portoinst-cpu_guarantee_cores_txxx,100)",
                        config.tag(),
                        "guarantee",
                        "#ff0000",
                        2,
                        false));
                chart.addSignal(
                    new GolovanSignal(
                        "div(sum(" + statsPrefix + "rusage-user_ammm,"
                        + statsPrefix + "rusage-system_ammm),"
                        + statsPrefix + "instance-alive_ammm)",
                        config.tag(),
                        "average",
                        null,
                        1,
                        false));
                chart.addSignal(
                    new GolovanSignal(
                        statsPrefix + "rusage-total_axxx",
                        config.tag(),
                        "max",
                        null,
                        1,
                        false));
                group.addChart(chart);
                panel.addCharts(
                    GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                    null,
                    group);
            }

            GolovanChartGroup portoUsageGroup = new GolovanChartGroup("", "");
            GolovanChart portoUsageChart = new GolovanChart(
                "porto-cpu-usage",
                " porto cpu usage (cores)",
                false,
                false,
                0d);
            portoUsageChart.addSignal(
                new GolovanSignal(
                    "portoinst-cpu_guarantee_cores_txxx",
                    config.tag(),
                    "guarantee",
                    "#ff0000",
                    2,
                    false));
            portoUsageChart.addSignal(
                new GolovanSignal(
                    "quant(portoinst-cpu_wait_slot_hgram, 99)",
                    config.tag(),
                    "wait",
                    "#ff8000",
                    2,
                    false));
            portoUsageChart.addSignal(
                new GolovanSignal(
                    "quant(portoinst-cpu_usage_slot_hgram, 99)",
                    config.tag(),
                    "usage",
                    null,
                    2,
                    false));
            portoUsageGroup.addChart(portoUsageChart);

            GolovanChart portoThreadsChart = new GolovanChart(
                "porto-threads",
                " total threads count",
                false,
                false,
                null);
            portoThreadsChart.addSignal(
                new GolovanSignal(
                    "portoinst-thread_count_tmmv",
                    config.tag(),
                    "threads",
                    null,
                    null,
                    false));
            portoUsageGroup.addChart(portoThreadsChart);

            panel.addCharts(
                GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                null,
                portoUsageGroup);

            if (cpuMonitor != null) {
                GolovanChartGroup group =
                    new GolovanChartGroup(statsPrefix, statsPrefix);
                GolovanChart uptime = new GolovanChart(
                    "uptime",
                    " cumulative uptime (seconds)",
                    false,
                    false,
                    null);
                uptime.addSplitSignal(
                    config,
                    statsPrefix + "uptime_ammm",
                    0,
                    false,
                    false);
                group.addChart(uptime);
                panel.addCharts(
                    GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                    null,
                    group);
            }
        }
    }

    private static class StaticStatsStater implements Stater {
        private final List<LongPair<String>> stats;

        StaticStatsStater(final List<LongPair<String>> stats) {
            this.stats = stats;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            for (LongPair<String> stat: stats) {
                statsConsumer.stat(stat.second(), stat.first());
            }
        }
    }

    private static class FreeSpaceStater implements Stater {
        private final Map<String, FileStore> signals;
        private final Logger logger;

        FreeSpaceStater(
            final Map<String, FileStore> signals,
            final Logger logger)
        {
            this.signals = signals;
            this.logger = logger;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            boolean failed = false;
            for (Map.Entry<String, FileStore> entry: signals.entrySet()) {
                String prefix = entry.getKey();
                long freeSpace;
                try {
                    freeSpace = entry.getValue().getUsableSpace();
                } catch (IOException e) {
                    failed = true;
                    freeSpace = 0L;
                    logger.log(
                        Level.SEVERE,
                        "Failed to get free space for " + prefix,
                        e);
                }
                statsConsumer.stat(prefix + "-free-space_annn", freeSpace);
                statsConsumer.stat(
                    prefix + "-no-free-space_ammx",
                    freeSpace == 0L);
            }
            statsConsumer.stat("free-space-stater-failed_ammx", failed);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            for (String path: signals.keySet()) {
                GolovanChart chart = new GolovanChart(
                    path + "-free-space",
                    " free space (bytes)",
                    false,
                    false,
                    0d);
                chart.addSignal(
                    new GolovanSignal(
                        statsPrefix + path + "-free-space_annn",
                        config.tag(),
                        "bytes",
                        null,
                        0,
                        false));
                group.addChart(chart);
            }
            panel.addCharts(
                GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                null,
                group);
        }
    }

    private class DiskVolumesStater implements Stater {
        private final List<String> diskVolumes;

        DiskVolumesStater(final List<String> diskVolumes) {
            this.diskVolumes = diskVolumes;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
        {
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            GolovanChart chart = new GolovanChart(
                "disk-volumes-usage",
                " disk volumes usage",
                false,
                false,
                0d);
            for (String diskVolume: diskVolumes) {
                chart.addSignal(
                    new GolovanSignal(
                        "portoinst-volume_" + diskVolume + "_usage_perc_txxx",
                        config.tag(),
                        diskVolume,
                        null,
                        1,
                        false));
            }
            group.addChart(chart);
            panel.addCharts(
                GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                null,
                group);
        }

        @Override
        public void addToAlertsConfig(
            final IniConfig alertsConfig,
            final ImmutableGolovanPanelConfig panelConfig,
            final String statsPrefix)
            throws BadRequestException
        {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            for (String diskVolume: diskVolumes) {
                alerts.createAlert(
                    alertsConfig,
                    alerts.module()
                    + '-'
                    + GolovanAlertsConfig.clearAlertName(diskVolume)
                    + "-free-space",
                    "portoinst-volume_" + diskVolume + "_usage_perc_txxx",
                    new AlertThresholds(90d, null, null));
            }
        }
    }

    private static class FilesStater implements Stater {
        private final Map<String, ImmutableFilesStaterConfig> filesStaters;
        private final Logger logger;

        FilesStater(
            final Map<String, ImmutableFilesStaterConfig> filesStaters,
            final Logger logger)
        {
            this.filesStaters = filesStaters;
            this.logger = logger;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int totalFailures = 0;
            long totalStart = TimeSource.INSTANCE.currentTimeMillis();
            for (Map.Entry<String, ImmutableFilesStaterConfig> entry
                : filesStaters.entrySet())
            {
                String name = entry.getKey();
                ImmutableFilesStaterConfig config = entry.getValue();
                FileStaterWalker walker =
                    new FileStaterWalker(
                        config.regexFilter(),
                        config.maxFiles());
                Path root = config.root();
                IOException e = null;
                long start = TimeSource.INSTANCE.currentTimeMillis();
                try {
                    Files.walkFileTree(root, walker);
                } catch (IOException ex) {
                    e = ex;
                }
                long time = TimeSource.INSTANCE.currentTimeMillis() - start;
                if (e == null && walker.e != null) {
                    e = walker.e;
                }
                if (e != null) {
                    logger.log(
                        Level.SEVERE,
                        "For file stater " + name
                        + " failed to walk file tree from root " + root
                        + ", failed at path: " + walker.failedPath,
                        e);
                }
                statsConsumer.stat(
                    name + "-files-count_ammx",
                    walker.filesCount);
                statsConsumer.stat(
                    name + "-files-size_ammx",
                    walker.filesSize);
                int failures;
                if (e == null) {
                    failures = 0;
                } else {
                    ++totalFailures;
                    failures = 1;
                }
                statsConsumer.stat(
                    name + "-files-stat-failures_ammx",
                    failures);
                statsConsumer.stat(name + "-files-stat-time_ammx", time);
            }
            long totalTime =
                TimeSource.INSTANCE.currentTimeMillis() - totalStart;
            statsConsumer.stat(
                "files-stat-total-failures_ammx",
                totalFailures);
            statsConsumer.stat("files-stat-total-time_ammx", totalTime);
        }
    }

    public static class FileStaterWalker implements FileVisitor<Path> {
        private final java.util.regex.Pattern regexFilter;
        private final int maxFiles;
        private IOException e = null;
        private Path failedPath = null;
        private int filesCount = 0;
        private long filesSize = 0L;

        public FileStaterWalker(
            final java.util.regex.Pattern regexFilter,
            final int maxFiles)
        {
            this.regexFilter = regexFilter;
            this.maxFiles = maxFiles;
        }

        @Override
        public FileVisitResult postVisitDirectory(
            final Path dir,
            final IOException e)
        {
            if (e == null) {
                return FileVisitResult.CONTINUE;
            } else {
                this.e = e;
                failedPath = dir;
                return FileVisitResult.TERMINATE;
            }
        }

        @Override
        public FileVisitResult preVisitDirectory(
            final Path dir,
            final BasicFileAttributes attrs)
        {
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFile(
            final Path file,
            final BasicFileAttributes attrs)
        {
            if (regexFilter == null
                || regexFilter.matcher(file.toString()).matches())
            {
                filesSize += attrs.size();
                if (++filesCount >= maxFiles) {
                    return FileVisitResult.TERMINATE;
                }
            }
            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(
            final Path file,
            final IOException e)
        {
            this.e = e;
            failedPath = file;
            return FileVisitResult.TERMINATE;
        }
    }

    private class InstanceAliveStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            statsConsumer.stat("instance-alive_ammm", 1L);
            statsConsumer.stat("instance-alive_ammn", 1L);
            statsConsumer.stat("instance-alive_ammv", 1L);
            statsConsumer.stat("instance-alive_ammx", 1L);
            statsConsumer.stat("instance-alive_ammt", 1L);
            long pingEnabled = booleanToLong(pingEnabled(null));
            statsConsumer.stat("ping-enabled_ammm", pingEnabled);
            statsConsumer.stat("ping-enabled_ammn", pingEnabled);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            GolovanChart chart = new GolovanChart(
                "instances-alive",
                " instances alive",
                false,
                true,
                null);
            chart.addSignal(
                new GolovanSignal(
                    "mul(" + statsPrefix + "instance-alive_ammm,5)",
                    config.tag(),
                    "average instances alive",
                    null,
                    2,
                    false));
            chart.addSignal(
                new GolovanSignal(
                    statsPrefix + "instance-alive_ammn",
                    config.tag(),
                    "min instances alive",
                    null,
                    0,
                    false));
            chart.addSignal(
                new GolovanSignal(
                    statsPrefix + "ping-enabled_ammn",
                    config.tag(),
                    "min ping enabled",
                    null,
                    0,
                    false));
            group.addChart(chart);
            panel.addCharts(
                GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                null,
                group);
        }
    }

    private enum HeapStater implements Stater {
        INSTANCE;

        @SuppressWarnings("ImmutableEnumChecker")
        private final MemoryMXBean mxBean =
            ManagementFactory.getMemoryMXBean();

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            MemoryUsage usage = mxBean.getHeapMemoryUsage();
            long used = usage.getUsed();
            statsConsumer.stat("heap-used_avvv", used);
            statsConsumer.stat("heap-used_ammx", used);
            statsConsumer.stat("heap-used_axxx", used);
            long committed = usage.getCommitted();
            statsConsumer.stat("heap-committed_avvv", committed);
            statsConsumer.stat("heap-committed_ammx", committed);
            statsConsumer.stat("heap-committed_axxx", committed);
            long max = usage.getMax();
            statsConsumer.stat("heap-max_avvv", max);
            statsConsumer.stat("heap-max_ammx", max);
            statsConsumer.stat("heap-max_axxx", max);
            if (max > 0L) {
                double usedPct = ((double) (used * PERCENTS)) / max;
                statsConsumer.stat("heap-used-pct_avvv", usedPct);
                statsConsumer.stat("heap-used-pct_axxx", usedPct);
                double committedPct = ((double) (committed * PERCENTS)) / max;
                statsConsumer.stat("heap-committed-pct_avvv", committedPct);
                statsConsumer.stat("heap-committed-pct_axxx", committedPct);
            }
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);

            GolovanChart absolutes = new GolovanChart(
                "heap-usage",
                " heap usage (bytes)",
                false,
                false,
                0d);
            absolutes.addSignal(
                new GolovanSignal(
                    statsPrefix + "heap-max_axxx",
                    config.tag(),
                    "max",
                    "#ff0000",
                    0,
                    false));
            absolutes.addSignal(
                new GolovanSignal(
                    statsPrefix + "heap-committed_axxx",
                    config.tag(),
                    "committed",
                    null,
                    0,
                    false));
            absolutes.addSignal(
                new GolovanSignal(
                    statsPrefix + "heap-used_axxx",
                    config.tag(),
                    "usage",
                    null,
                    0,
                    false));
            group.addChart(absolutes);

            GolovanChart pct = new GolovanChart(
                "heap-usage-pct",
                " heap usage (%)",
                false,
                false,
                0d);
            pct.addSignal(
                new GolovanSignal(
                    "const(100)",
                    config.tag(),
                    "100%",
                    "#ff0000",
                    0,
                    false));
            pct.addSignal(
                new GolovanSignal(
                    "const(75)",
                    config.tag(),
                    "75%",
                    "#ff8000",
                    0,
                    false));
            pct.addSignal(
                new GolovanSignal(
                    statsPrefix + "heap-committed-pct_axxx",
                    config.tag(),
                    "committed",
                    null,
                    0,
                    false));
            pct.addSignal(
                new GolovanSignal(
                    statsPrefix + "heap-used-pct_axxx",
                    config.tag(),
                    "usage",
                    null,
                    0,
                    false));
            group.addChart(pct);

            panel.addCharts(
                GolovanPanelConfig.CATEGORY_MEMORY,
                null,
                group);
        }
    }

    private static class IncomingConnectionsStater implements Stater {
        private final TimeFrameQueue<Object> incomingConnections;
        private final TimeFrameQueue<Object> limitedConnections;
        private final double normCoeff;

        IncomingConnectionsStater(
            final TimeFrameQueue<Object> incomingConnections,
            final TimeFrameQueue<Object> limitedConnections,
            final double normCoeff)
        {
            this.incomingConnections = incomingConnections;
            this.limitedConnections = limitedConnections;
            this.normCoeff = normCoeff;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int size = Iterators.size(incomingConnections.iterator());
            statsConsumer.stat("incoming-connections_axxx", size * normCoeff);
            statsConsumer.stat("incoming-connections_ammm", size);
            size = Iterators.size(limitedConnections.iterator());
            statsConsumer.stat("limited-connections_axxx", size * normCoeff);
            statsConsumer.stat("limited-connections_ammm", size);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);

            GolovanChart incoming = new GolovanChart(
                "incoming-connections",
                " incoming connections (rps)",
                false,
                true,
                0d);
            incoming.addSignal(
                new GolovanSignal(
                    statsPrefix + "incoming-connections_axxx",
                    config.tag(),
                    "max per instance",
                    null,
                    1,
                    false));
            incoming.addSignal(
                new GolovanSignal(
                    statsPrefix + "incoming-connections_ammm",
                    config.tag(),
                    "average for cluster",
                    null,
                    1,
                    false));
            group.addChart(incoming);

            GolovanChart limited = new GolovanChart(
                "limited-connections",
                " limited connections (rps)",
                false,
                true,
                0d);
            limited.addSignal(
                new GolovanSignal(
                    statsPrefix + "limited-connections_axxx",
                    config.tag(),
                    "max per instance",
                    null,
                    1,
                    false));
            limited.addSignal(
                new GolovanSignal(
                    statsPrefix + "limited-connections_ammm",
                    config.tag(),
                    "average for cluster",
                    null,
                    1,
                    false));
            group.addChart(limited);

            panel.addCharts(
                GolovanPanelConfig.CATEGORY_CONNECTIONS,
                null,
                group);
        }
    }

    private static class ConnectionsLimiterStater implements Stater {
        private final Limiter connectionsLimiter;

        ConnectionsLimiterStater(final Limiter connectionsLimiter) {
            this.connectionsLimiter = connectionsLimiter;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int activeConnections = connectionsLimiter.concurrentRequests();
            statsConsumer.stat("active-connections_axxx", activeConnections);
            statsConsumer.stat("active-connections_ammx", activeConnections);
            statsConsumer.stat("active-connections_ammm", activeConnections);
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            GolovanChart chart = new GolovanChart(
                "active-connections",
                " active connections",
                false,
                true,
                0d);
            chart.addSignal(
                new GolovanSignal(
                    statsPrefix + "active-connections_axxx",
                    config.tag(),
                    "max",
                    null,
                    0,
                    false));
            chart.addSignal(
                new GolovanSignal(
                    "div(" + statsPrefix + "active-connections_ammm,"
                    + statsPrefix + "instance-alive_ammm)",
                    config.tag(),
                    "average",
                    null,
                    0,
                    false));
            group.addChart(chart);
            panel.addCharts(
                GolovanPanelConfig.CATEGORY_CONNECTIONS,
                null,
                group);
        }
    }

    private class HealthStater implements Stater {
        private final long certificateValidUntil;

        HealthStater(final long certificateValidUntil) {
            this.certificateValidUntil = certificateValidUntil;
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            if (serviceContextRenewalTask != null) {
                Tvm2ServiceContextRenewalTask.Status status =
                    serviceContextRenewalTask.status();
                long expiring =
                    booleanToLong(
                        status != Tvm2ServiceContextRenewalTask.Status.OK);
                statsConsumer.stat("tvm-keys-expiring_ammx", expiring);
            }

            if (certificateValidUntil < Long.MAX_VALUE) {
                long ttl =
                    certificateValidUntil
                    - TimeSource.INSTANCE.currentTimeMillis();
                statsConsumer.stat("certificate-ttl_annn", ttl);
                statsConsumer.stat(
                    "certificate-about-to-expire_ammx",
                    booleanToLong(ttl <= 604800000L));
            }

            for (HttpChecker httpChecker: httpCheckers) {
                statsConsumer.stat(
                    httpChecker.name() + "-check-failed_ammx",
                    booleanToLong(!httpChecker.ok()));
            }

            ExternalDataUpdater[] externalDataUpdaters =
                AbstractHttpServer.this.externalDataUpdaters;
            for (ExternalDataUpdater updater: externalDataUpdaters) {
                // Preserving TimeSource invocation ordering, so we don't get
                // negative data age
                long lastUpdate = updater.lastUpdate();
                statsConsumer.stat(
                    updater.name() + "-data-age_axxx",
                    TimeSource.INSTANCE.currentTimeMillis() - lastUpdate);
            }
        }

        @Override
        public void addToGolovanPanel(
            final GolovanPanel panel,
            final String statsPrefix)
        {
            ImmutableGolovanPanelConfig config = panel.config();
            GolovanChartGroup group =
                new GolovanChartGroup(statsPrefix, statsPrefix);
            boolean empty = true;
            GolovanChart chart = new GolovanChart(
                "checks-status",
                " checks status",
                false,
                true,
                null);
            if (serviceContextRenewalTask != null) {
                empty = false;
                chart.addSignal(
                    new GolovanSignal(
                        statsPrefix + "tvm-keys-expiring_ammx",
                        config.tag(),
                        "TVM keys are expiring",
                        null,
                        0,
                        false));
            }

            if (certificateValidUntil < Long.MAX_VALUE) {
                empty = false;
                chart.addSignal(
                    new GolovanSignal(
                        statsPrefix + "certificate-about-to-expire_ammx",
                        config.tag(),
                        "SSL certificate is about to expire",
                        null,
                        0,
                        false));
            }

            for (HttpChecker httpChecker: httpCheckers) {
                empty = false;
                String name = httpChecker.name();
                chart.addSignal(
                    new GolovanSignal(
                        statsPrefix + name + "-check-failed_ammx",
                        config.tag(),
                        name + " check failed",
                        null,
                        0,
                        false));
            }
            if (!empty) {
                group.addChart(chart);
            }

            chart = new GolovanChart(
                "external-data-age",
                " data age (ms)",
                false,
                true,
                null);
            boolean emptyUpdaters = true;
            ExternalDataUpdater[] externalDataUpdaters =
                AbstractHttpServer.this.externalDataUpdaters;
            for (ExternalDataUpdater updater: externalDataUpdaters) {
                emptyUpdaters = false;
                String name = updater.name();
                chart.addSignal(
                    new GolovanSignal(
                        statsPrefix + name + "-data-age_axxx",
                        config.tag(),
                        name,
                        null,
                        0,
                        false));
            }

            if (!emptyUpdaters) {
                empty = false;
                group.addChart(chart);
            }

            if (!empty) {
                panel.addCharts(
                    GolovanPanelConfig.CATEGORY_SYSTEM_INFO,
                    null,
                    group);
            }
        }

        @Override
        public void addToAlertsConfig(
            final IniConfig alertsConfig,
            final ImmutableGolovanPanelConfig panelConfig,
            final String statsPrefix)
            throws BadRequestException
        {
            ImmutableGolovanAlertsConfig alerts = panelConfig.alerts();
            if (serviceContextRenewalTask != null) {
                alerts.createAlert(
                    alertsConfig,
                    alerts.module() + "-tvm-keys-expiring",
                    statsPrefix + "tvm-keys-expiring_ammx",
                    new AlertThresholds(0.5d, null, null));
            }

            if (certificateValidUntil < Long.MAX_VALUE) {
                alerts.createAlert(
                    alertsConfig,
                    alerts.module() + "-certificate-expiring",
                    statsPrefix + "certificate-about-to-expire_ammx",
                    new AlertThresholds(0.5d, null, null));
            }

            ExternalDataUpdater[] externalDataUpdaters =
                AbstractHttpServer.this.externalDataUpdaters;
            for (ExternalDataUpdater updater: externalDataUpdaters) {
                String name = updater.name();
                alerts.createAlert(
                    alertsConfig,
                    alerts.module() + '-' + name + "-data-too-old",
                    statsPrefix + name + "-data-age_axxx",
                    new AlertThresholds(
                        updater.updateInterval() * 5d,
                        null,
                        null));
            }
        }
    }

    public enum UncaughtExceptionHandler
        implements Thread.UncaughtExceptionHandler
    {
        INSTANCE;

        private static final int BUF_SIZE = 8192;

        private final Encoder encoder = new Encoder(
            StandardCharsets.UTF_8.newEncoder()
                .onMalformedInput(CodingErrorAction.REPLACE)
                .onUnmappableCharacter(CodingErrorAction.REPLACE));
        private final StringBuilder sb = new StringBuilder(BUF_SIZE);
        private final StringBuilderWriter sbw = new StringBuilderWriter(sb);
        private char[] cbuf = new char[BUF_SIZE];
        private byte[] buf = new byte[BUF_SIZE];
        @SuppressWarnings("UnusedVariable")
        private byte[] deadWeight = new byte[262144];

        UncaughtExceptionHandler() {
            // Init encoder and precallocate its intenal buffers
            try {
                encoder.process(cbuf);
            } catch (CharacterCodingException e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public synchronized void uncaughtException(
            final Thread thread,
            final Throwable cause)
        {
            // drop the dead weight
            deadWeight = null;
            sb.setLength(0);
            sb.append("Exception in Thread[");
            sb.append(thread.getName());
            ThreadGroup group = thread.getThreadGroup();
            if (group != null) {
                sb.append(',');
                sb.append(group.getName());
            }
            sb.append("]: ");
            cause.printStackTrace(sbw);

            int len = sb.length();
            if (len > cbuf.length) {
                try {
                    cbuf = new char[len];
                } catch (Throwable t) {
                    len = cbuf.length;
                }
            }
            sb.getChars(0, len, cbuf, 0);
            try {
                encoder.process(cbuf, 0, len);
                encoder.processWith(
                    new OutputStreamProcessorAdapter(System.err));
            } catch (Throwable t) {
                len = Math.min(buf.length, len);
                for (int i = 0; i < len; ++i) {
                    char c = cbuf[i];
                    if (c < 128) {
                        buf[i] = (byte) c;
                    } else {
                        buf[i] = '?';
                    }
                }
                System.err.write(buf, 0, len);
            }
            System.err.flush();
            Runtime.getRuntime().halt(1);
        }
    }

    public static <C extends ImmutableBaseServerConfig, T> HttpServer<C, T>
        main(
            final HttpServerFactory<C, T> serverFactory,
            final String... args)
            throws ConfigException, IOException
    {
        if (args.length == 1) {
            final PrintStream out = System.out;
            final PrintStream err = System.err;
            IniConfig iniConfig = new IniConfig(Paths.get(args[0]));
            final HttpServer<C, T> server = serverFactory.create(iniConfig);
            ImmutableLoggersConfig config = server.config().loggers();
            if (config.stdout() != null) {
                System.setOut(
                    new PrintStream(
                        new LoggerOutputStream(
                            config.loggerForStdout(),
                            config.stdout().charset()),
                        true,
                        config.stdout().charset().name()));
            }
            if (config.stderr() != null) {
                System.setErr(
                    new PrintStream(
                        new LoggerOutputStream(
                            config.loggerForStderr(),
                            config.stderr().charset()),
                        true,
                        config.stderr().charset().name()));
            }
            iniConfig.checkUnusedKeys();
            server.start();
            Runtime.getRuntime().addShutdownHook(new Thread() {
                @Override
                public void run() {
                    try {
                        server.close();
                        server.join();
                    } catch (InterruptedException e) {
                    } catch (IOException e) {
                        throw new RuntimeException("Server stop failed", e);
                    } finally {
                        System.setOut(out);
                        System.setErr(err);
                    }
                    server.logger().info("Shutdown hook completed");
                }
            });
            System.out.println("###started###");
            return server;
        } else {
            System.err.println(
                "Usage: " + serverFactory.name() + " <config file>");
            return null;
        }
    }

    public static void main(final String... args)
        throws ConfigException, IOException
    {
        if (main(new DefaultHttpServerFactory<>(), args) == null) {
            System.exit(1);
        }
    }
}

