package ru.yandex.http.util.server;

import java.io.File;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.Header;

import ru.yandex.collection.LongPair;
import ru.yandex.http.config.AbstractHttpConnectionConfigBuilder;
import ru.yandex.http.config.ExternalDataConfig;
import ru.yandex.http.config.ExternalDataConfigBuilder;
import ru.yandex.http.config.ExternalDataConfigDefaults;
import ru.yandex.http.config.ServerHttpsConfig;
import ru.yandex.http.config.ServerHttpsConfigBuilder;
import ru.yandex.http.config.URICheckConfig;
import ru.yandex.http.config.URICheckConfigBuilder;
import ru.yandex.http.config.URICheckConfigDefaults;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeIntegerValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;

public abstract class AbstractHttpServerConfigBuilder
    <T extends AbstractHttpServerConfigBuilder<T>>
    extends AbstractHttpConnectionConfigBuilder<T>
    implements HttpServerConfig
{
    public static final String BUILT_DATE = YandexHeaders.BUILT_DATE;

    private static final int PERCENTS = 100;
    private static final int CONNECTIONS_SCALE = 10;
    private static final CollectionParser<String, Set<String>, Exception>
        TRIMMED_STRING_SET_PARSER = new CollectionParser<>(
            NonEmptyValidator.TRIMMED,
            HashSet::new);
    private static final CollectionParser<Integer, Set<Integer>, Exception>
        KEEP_ALIVE_STATUSES_PARSER = new CollectionParser<>(
            NonEmptyValidator.TRIMMED.andThen(Integer::valueOf),
            HashSet::new);

    private String name;
    private String origin;
    private int port;
    private int workers;
    private int timeout;
    private long connectionCloseCheckInterval;
    private int connections;
    private int maxGarbageConnections;
    private int minConnectionsPerHost;
    private int maxConnectionsPerHost;
    private boolean rejectConnectionsOverLimit;
    private int timerResolution;
    private int backlog;
    private int linger;
    private boolean gzip;
    private boolean lazyBind;
    private boolean pingEnabledOnStartup;
    private boolean cpuStater;
    private boolean memoryStater;
    private boolean gcStater;
    private boolean heapStater;
    private boolean instanceAliveStater;
    private Set<String> hiddenHeaders;
    private List<Header> staticHeaders;
    private List<LongPair<String>> staticStats;
    private Map<String, Set<String>> statsAliases;
    private String statsPrefix;
    private boolean keepUnprefixedStats;
    private Map<String, FileStore> freeSpaceSignals;
    private Map<String, FilesStaterConfigBuilder> filesStaters;
    private ServerHttpsConfigBuilder httpsConfig;
    private Set<String> debugFlags;
    private Set<Integer> keepAliveStatuses;
    private Set<String> defaultHttpMethods;
    private Map<String, URICheckConfigBuilder> httpChecks;
    private Path localCacheDir;
    private Map<String, ExternalDataConfigBuilder> externalData;

    protected AbstractHttpServerConfigBuilder(final HttpServerConfig config) {
        super(config);
        name(config.name());
        origin(config.origin());
        port(config.port());
        workers(config.workers());
        timeout(config.timeout());
        connectionCloseCheckInterval(config.connectionCloseCheckInterval());
        connections(config.connections());
        maxGarbageConnections(config.maxGarbageConnections());
        minConnectionsPerHost(config.minConnectionsPerHost());
        maxConnectionsPerHost(config.maxConnectionsPerHost());
        rejectConnectionsOverLimit(config.rejectConnectionsOverLimit());
        timerResolution(config.timerResolution());
        backlog(config.backlog());
        linger(config.linger());
        gzip(config.gzip());
        lazyBind(config.lazyBind());
        pingEnabledOnStartup(config.pingEnabledOnStartup());
        cpuStater(config.cpuStater());
        memoryStater(config.memoryStater());
        gcStater(config.gcStater());
        heapStater(config.heapStater());
        instanceAliveStater(config.instanceAliveStater());
        hiddenHeaders(config.hiddenHeaders());
        staticHeaders(config.staticHeaders());
        staticStats(config.staticStats());
        statsAliases(config.statsAliases());
        statsPrefix(config.statsPrefix());
        keepUnprefixedStats(config.keepUnprefixedStats());
        freeSpaceSignals(config.freeSpaceSignals());
        filesStaters(config.filesStaters());
        httpsConfig(config.httpsConfig());
        debugFlags(config.debugFlags());
        keepAliveStatuses(config.keepAliveStatuses());
        defaultHttpMethods(config.defaultHttpMethods());
        httpChecks(config.httpChecks());
        localCacheDir(config.localCacheDir());
        externalData(config.externalData());
    }

    protected AbstractHttpServerConfigBuilder(
        final IniConfig config,
        final HttpServerConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        name = defaults.name();
        origin = defaults.origin();
        port = config.get(
            "port",
            defaults.port(),
            NonNegativeIntegerValidator.INSTANCE);
        int min = config.get(
            "workers.min",
            defaults.workers(),
            PositiveIntegerValidator.INSTANCE);
        int percent = config.get(
            "workers.percent",
            0,
            NonNegativeIntegerValidator.INSTANCE);
        workers = Math.max(
            min,
            (Runtime.getRuntime().availableProcessors() * percent) / PERCENTS);
        timeout = config.getIntegerDuration("timeout", defaults.timeout());
        connectionCloseCheckInterval = config.getLongDuration(
            "connection-close-check-interval",
            defaults.connectionCloseCheckInterval());
        connections = config.get(
            "connections",
            defaults.connections(),
            PositiveIntegerValidator.INSTANCE);
        maxGarbageConnections =
            Math.min(
                config.get(
                    "max-garbage-connections",
                    defaults.maxGarbageConnections(),
                    PositiveIntegerValidator.INSTANCE),
                Math.max(1, connections / CONNECTIONS_SCALE));
        minConnectionsPerHost = config.get(
            "min-connections-per-host",
            defaults.minConnectionsPerHost(),
            NonNegativeIntegerValidator.INSTANCE);
        maxConnectionsPerHost = config.get(
            "max-connections-per-host",
            defaults.maxConnectionsPerHost(),
            PositiveIntegerValidator.INSTANCE);
        rejectConnectionsOverLimit = config.getBoolean(
            "reject-connections-over-limit",
            defaults.rejectConnectionsOverLimit());

        timerResolution = config.getIntegerDuration(
            "timer.resolution",
            defaults.timerResolution());
        backlog = config.get(
            "backlog",
            defaults.backlog(),
            NonNegativeIntegerValidator.INSTANCE);
        linger = config.getInt("linger", defaults.linger());
        gzip = config.getBoolean("gzip", defaults.gzip());
        lazyBind = config.getBoolean("lazy-bind", defaults.lazyBind());
        pingEnabledOnStartup = config.getBoolean(
            "ping-enabled-on-startup",
            defaults.pingEnabledOnStartup());
        cpuStater = config.getBoolean("cpu-stater", defaults.cpuStater());
        memoryStater =
            config.getBoolean("memory-stater", defaults.memoryStater());
        gcStater = config.getBoolean("gc-stater", defaults.gcStater());
        heapStater = config.getBoolean("heap-stater", defaults.heapStater());
        instanceAliveStater = config.getBoolean(
            "instance-alive-stater",
            defaults.instanceAliveStater());
        hiddenHeaders = config.get(
            "hidden-headers",
            defaults.hiddenHeaders(),
            TRIMMED_STRING_SET_PARSER);

        staticHeaders = new ArrayList<>();
        IniConfig staticHeadersSection = config.section("static-headers");
        for (String header: staticHeadersSection.keys()) {
            staticHeaders.add(
                HeaderUtils.createHeader(
                    header,
                    staticHeadersSection.getString(header)));
        }

        staticStats = new ArrayList<>();
        IniConfig staticStatsSection = config.section("static-stats");
        for (String stat: staticStatsSection.keys()) {
            staticStats.add(
                new LongPair<>(staticStatsSection.getLong(stat), stat));
        }

        statsAliases = new HashMap<>();
        IniConfig statsAliasesSection = config.sectionOrNull("stats-aliases");
        if (statsAliasesSection == null) {
            statsAliases = new HashMap<>(defaults.statsAliases());
        } else {
            for (String stat: statsAliasesSection.keys()) {
                statsAliases.put(
                    stat,
                    statsAliasesSection.get(stat, TRIMMED_STRING_SET_PARSER));
            }
        }
        statsPrefix = config.get(
            "stats-prefix",
            defaults.statsPrefix(),
            NonEmptyValidator.INSTANCE);
        keepUnprefixedStats = config.getBoolean(
            "keep-unprefixed-stats",
            defaults.keepUnprefixedStats());

        IniConfig freeSpaceSignalsSection =
            config.section("free-space-signals");
        Set<String> keys = freeSpaceSignalsSection.keys();
        freeSpaceSignals = new LinkedHashMap<>(keys.size() << 1);
        for (String signal: freeSpaceSignalsSection.keys()) {
            freeSpaceSignals.put(
                signal,
                freeSpaceSignalsSection.get(
                    signal,
                    path -> Files.getFileStore(Paths.get(path))));
        }

        IniConfig filesStatersSection = config.section("files-staters");
        Set<String> names = filesStatersSection.sections().keySet();
        filesStaters = new LinkedHashMap<>(names.size() << 1);
        for (String name: names) {
            filesStaters.put(
                name,
                HttpServerConfig.filesStaterParser(name)
                    .extract(
                        filesStatersSection,
                        FilesStaterConfigDefaults.INSTANCE));
        }

        IniConfig httpsSection = config.sectionOrNull("https");
        ServerHttpsConfig httpsConfigDefaults = defaults.httpsConfig();
        if (httpsSection == null) {
            httpsConfig(httpsConfigDefaults);
        } else if (httpsConfigDefaults == null) {
            httpsConfig = new ServerHttpsConfigBuilder(httpsSection);
        } else {
            httpsConfig = new ServerHttpsConfigBuilder(
                httpsSection,
                httpsConfigDefaults);
        }

        debugFlags = config.get(
            "debug-flags",
            defaults.debugFlags(),
            TRIMMED_STRING_SET_PARSER);

        keepAliveStatuses = config.get(
            "keep-alive-statuses",
            defaults.keepAliveStatuses(),
            KEEP_ALIVE_STATUSES_PARSER);

        defaultHttpMethods =
            config.getAll(
                "default-http-methods",
                defaults.defaultHttpMethods(),
                TRIMMED_STRING_SET_PARSER);

        IniConfig httpChecksSection = config.section("http-check");
        names = httpChecksSection.sections().keySet();
        httpChecks = new LinkedHashMap<>(names.size() << 1);
        for (String name: names) {
            httpChecks.put(
                name,
                HttpServerConfig.httpCheckParser(name)
                    .extract(
                        httpChecksSection,
                        URICheckConfigDefaults.INSTANCE));
        }

        File localCacheDir = config.getDir("local-cache-dir", null);
        if (localCacheDir == null) {
            this.localCacheDir = null;
        } else {
            this.localCacheDir = localCacheDir.toPath();
        }

        IniConfig externalDataSection = config.section("external-data");
        names = externalDataSection.sections().keySet();
        externalData = new LinkedHashMap<>(names.size() << 1);
        for (String name: names) {
            externalData.put(
                name,
                HttpServerConfig.externalDataConfigParser(name)
                    .extract(
                        externalDataSection,
                        ExternalDataConfigDefaults.INSTANCE));
        }
    }

    @Override
    public String name() {
        return name;
    }

    public T name(final String name) {
        this.name = name;
        origin = name + ' ' + BUILT_DATE;
        return self();
    }

    @Override
    public String origin() {
        return origin;
    }

    public T origin(final String origin) {
        this.origin = origin;
        return self();
    }

    @Override
    public int port() {
        return port;
    }

    public T port(final int port) {
        this.port = port;
        return self();
    }

    @Override
    public int workers() {
        return workers;
    }

    public T workers(final int workers) {
        this.workers = workers;
        return self();
    }

    @Override
    public int timeout() {
        return timeout;
    }

    public T timeout(final int timeout) {
        this.timeout = timeout;
        return self();
    }

    @Override
    public long connectionCloseCheckInterval() {
        return connectionCloseCheckInterval;
    }

    public T connectionCloseCheckInterval(final long connectionCloseCheckInterval) {
        this.connectionCloseCheckInterval = connectionCloseCheckInterval;
        return self();
    }

    @Override
    public int connections() {
        return connections;
    }

    public T connections(final int connections) {
        this.connections = connections;
        return self();
    }

    @Override
    public int maxGarbageConnections() {
        return maxGarbageConnections;
    }

    public T maxGarbageConnections(final int maxGarbageConnections) {
        this.maxGarbageConnections = maxGarbageConnections;
        return self();
    }

    @Override
    public int minConnectionsPerHost() {
        return minConnectionsPerHost;
    }

    public T minConnectionsPerHost(final int minConnectionsPerHost) {
        this.minConnectionsPerHost = minConnectionsPerHost;
        return self();
    }

    @Override
    public int maxConnectionsPerHost() {
        return maxConnectionsPerHost;
    }

    public T maxConnectionsPerHost(final int maxConnectionsPerHost) {
        this.maxConnectionsPerHost = maxConnectionsPerHost;
        return self();
    }

    @Override
    public boolean rejectConnectionsOverLimit() {
        return rejectConnectionsOverLimit;
    }

    public T rejectConnectionsOverLimit(
        final boolean rejectConnectionsOverLimit)
    {
        this.rejectConnectionsOverLimit = rejectConnectionsOverLimit;
        return self();
    }

    @Override
    public int timerResolution() {
        return timerResolution;
    }

    public T timerResolution(final int timerResolution) {
        this.timerResolution = timerResolution;
        return self();
    }

    @Override
    public int backlog() {
        return backlog;
    }

    public T backlog(final int backlog) {
        this.backlog = backlog;
        return self();
    }

    @Override
    public int linger() {
        return linger;
    }

    public T linger(final int linger) {
        this.linger = linger;
        return self();
    }

    @Override
    public boolean gzip() {
        return gzip;
    }

    public T gzip(final boolean gzip) {
        this.gzip = gzip;
        return self();
    }

    @Override
    public boolean lazyBind() {
        return lazyBind;
    }

    public T lazyBind(final boolean lazyBind) {
        this.lazyBind = lazyBind;
        return self();
    }

    @Override
    public boolean pingEnabledOnStartup() {
        return pingEnabledOnStartup;
    }

    public T pingEnabledOnStartup(final boolean pingEnabledOnStartup) {
        this.pingEnabledOnStartup = pingEnabledOnStartup;
        return self();
    }

    @Override
    public boolean cpuStater() {
        return cpuStater;
    }

    public T cpuStater(final boolean cpuStater) {
        this.cpuStater = cpuStater;
        return self();
    }

    @Override
    public boolean memoryStater() {
        return memoryStater;
    }

    public T memoryStater(final boolean memoryStater) {
        this.memoryStater = memoryStater;
        return self();
    }

    @Override
    public boolean gcStater() {
        return gcStater;
    }

    public T gcStater(final boolean gcStater) {
        this.gcStater = gcStater;
        return self();
    }

    @Override
    public boolean heapStater() {
        return heapStater;
    }

    public T heapStater(final boolean heapStater) {
        this.heapStater = heapStater;
        return self();
    }

    @Override
    public boolean instanceAliveStater() {
        return instanceAliveStater;
    }

    public T instanceAliveStater(final boolean instanceAliveStater) {
        this.instanceAliveStater = instanceAliveStater;
        return self();
    }

    @Override
    public Set<String> hiddenHeaders() {
        return hiddenHeaders;
    }

    public T hiddenHeaders(final Set<String> hiddenHeaders) {
        this.hiddenHeaders = hiddenHeaders;
        return self();
    }

    @Override
    public List<Header> staticHeaders() {
        return staticHeaders;
    }

    public T staticHeaders(final List<Header> staticHeaders) {
        this.staticHeaders = new ArrayList<>(staticHeaders);
        return self();
    }

    @Override
    public List<LongPair<String>> staticStats() {
        return staticStats;
    }

    public T staticStats(final List<LongPair<String>> staticStats) {
        this.staticStats = new ArrayList<>(staticStats);
        return self();
    }

    @Override
    public Map<String, Set<String>> statsAliases() {
        return statsAliases;
    }

    public T statsAliases(final Map<String, Set<String>> statsAliases) {
        this.statsAliases = new HashMap<>(statsAliases.size() << 1);
        for (Map.Entry<String, Set<String>> entry: statsAliases.entrySet()) {
            this.statsAliases.put(
                entry.getKey(),
                new HashSet<>(entry.getValue()));
        }
        return self();
    }

    @Override
    public String statsPrefix() {
        return statsPrefix;
    }

    public T statsPrefix(final String statsPrefix) {
        this.statsPrefix = statsPrefix;
        return self();
    }

    @Override
    public boolean keepUnprefixedStats() {
        return keepUnprefixedStats;
    }

    public T keepUnprefixedStats(final boolean keepUnprefixedStats) {
        this.keepUnprefixedStats = keepUnprefixedStats;
        return self();
    }

    @Override
    public Map<String, FileStore> freeSpaceSignals() {
        return freeSpaceSignals;
    }

    public T freeSpaceSignals(final Map<String, FileStore> freeSpaceSignals) {
        this.freeSpaceSignals =
            new LinkedHashMap<>(freeSpaceSignals.size() << 1);
        for (Map.Entry<String, FileStore> entry: freeSpaceSignals.entrySet()) {
            this.freeSpaceSignals.put(entry.getKey(), entry.getValue());
        }
        return self();
    }

    @Override
    public Map<String, FilesStaterConfigBuilder> filesStaters() {
        return filesStaters;
    }

    public T filesStaters(
        final Map<String, ? extends FilesStaterConfig> filesStaters)
    {
        this.filesStaters = new LinkedHashMap<>(filesStaters.size() << 1);
        for (Map.Entry<String, ? extends FilesStaterConfig> entry
            : filesStaters.entrySet())
        {
            this.filesStaters.put(
                entry.getKey(),
                new FilesStaterConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public ServerHttpsConfigBuilder httpsConfig() {
        return httpsConfig;
    }

    public T httpsConfig(final ServerHttpsConfig httpsConfig) {
        if (httpsConfig == null) {
            this.httpsConfig = null;
        } else {
            this.httpsConfig = new ServerHttpsConfigBuilder(httpsConfig);
        }
        return self();
    }

    @Override
    public Set<String> debugFlags() {
        return debugFlags;
    }

    public T debugFlags(final Set<String> debugFlags) {
        this.debugFlags = new HashSet<>(debugFlags);
        return self();
    }

    @Override
    public Set<Integer> keepAliveStatuses() {
        return keepAliveStatuses;
    }

    public T keepAliveStatuses(final Set<Integer> keepAliveStatuses) {
        this.keepAliveStatuses = new HashSet<>(keepAliveStatuses);
        return self();
    }

    @Override
    public Set<String> defaultHttpMethods() {
        return defaultHttpMethods;
    }

    public T defaultHttpMethods(final Set<String> defaultHttpMethods) {
        this.defaultHttpMethods = new LinkedHashSet<>(defaultHttpMethods);
        return self();
    }

    @Override
    public Map<String, URICheckConfigBuilder> httpChecks() {
        return httpChecks;
    }

    public T httpChecks(
        final Map<String, ? extends URICheckConfig> httpChecks)
    {
        this.httpChecks = new LinkedHashMap<>(httpChecks.size() << 1);
        for (Map.Entry<String, ? extends URICheckConfig> entry
            : httpChecks.entrySet())
        {
            this.httpChecks.put(
                entry.getKey(),
                new URICheckConfigBuilder(entry.getValue()));
        }
        return self();
    }

    @Override
    public Path localCacheDir() {
        return localCacheDir;
    }

    public T localCacheDir(final Path localCacheDir) {
        this.localCacheDir = localCacheDir;
        return self();
    }

    @Override
    public Map<String, ExternalDataConfigBuilder> externalData() {
        return externalData;
    }

    public T externalData(
        final Map<String, ? extends ExternalDataConfig> externalData)
    {
        this.externalData = new LinkedHashMap<>(externalData.size() << 1);
        for (Map.Entry<String, ? extends ExternalDataConfig> entry
            : externalData.entrySet())
        {
            this.externalData.put(
                entry.getKey(),
                new ExternalDataConfigBuilder(entry.getValue()));
        }
        return self();
    }

    public void copyTo(final AbstractHttpServerConfigBuilder<?> other) {
        copyTo((AbstractHttpConnectionConfigBuilder<?>) other);
        other
            .name(name())
            .origin(origin())
            .port(port())
            .workers(workers())
            .connections(connections())
            .timeout(timeout())
            .connectionCloseCheckInterval(connectionCloseCheckInterval())
            .timerResolution(timerResolution())
            .backlog(backlog())
            .linger(linger())
            .gzip(gzip())
            .staticHeaders(staticHeaders())
            .staticStats(staticStats())
            .statsAliases(statsAliases())
            .cpuStater(cpuStater())
            .memoryStater(memoryStater())
            .gcStater(gcStater())
            .freeSpaceSignals(freeSpaceSignals())
            .filesStaters(filesStaters())
            .httpsConfig(httpsConfig())
            .debugFlags(debugFlags())
            .keepAliveStatuses(keepAliveStatuses())
            .defaultHttpMethods(defaultHttpMethods())
            .httpChecks(httpChecks())
            .localCacheDir(localCacheDir())
            .externalData(externalData);
    }
}

