package ru.yandex.http.util.server;

import java.io.IOException;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.ContentType;
import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.client.tvm2.Tvm2ClientConfigBuilder;
import ru.yandex.client.tvm2.Tvm2ServiceContextRenewalTask;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.concurrent.CloseableTimerTask;
import ru.yandex.concurrent.TimerTaskCloser;
import ru.yandex.function.ByteArrayProcessable;
import ru.yandex.function.Processable;
import ru.yandex.function.SimpleGenericAutoCloseableHolder;
import ru.yandex.http.config.ImmutableExternalDataConfig;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.ByteArrayProcessableWithContentType;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HeaderUtils;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.io.PathWriterProcessor;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.timesource.TimeSource;

public class ExternalDataUpdater extends CloseableTimerTask {
    private static final String CONTENT_TYPE_ATTRIBUTE = "user:content-type";
    private static final String CONTENT_ENCODING_ATTRIBUTE =
        "user:content-encoding";
    private static final ExternalDataSubscriber[] EMPTY_SUBSCRIBERS =
        new ExternalDataSubscriber[0];
    private static final long TVM_TICKETS_RENEWAL_INTERVAL = 600000L;

    private final String name;
    private final Path localCache;
    private final Logger logger;
    private final String uri;
    private final CloseableHttpClient client;
    private final long updateInterval;
    private final List<Map.Entry<String, String>> tvm2Headers;
    private final int tvm2HeadersSize;
    private final Tvm2TicketRenewalTask ticketRenewalTask;
    private volatile ExternalDataSubscriber[] subscribers = EMPTY_SUBSCRIBERS;
    private volatile long lastUpdate;

    public ExternalDataUpdater(
        final String name,
        final HttpServer<?, ?> server,
        final Tvm2ServiceContextRenewalTask serviceContextRenewalTask,
        final ImmutableExternalDataConfig config)
        throws ConfigException, HttpException, IOException
    {
        try (SimpleGenericAutoCloseableHolder<IOException> holder =
                new SimpleGenericAutoCloseableHolder<>(closeChain))
        {
            this.name = name;
            Path localCacheDir = server.config().localCacheDir();
            if (localCacheDir == null) {
                localCache = null;
            } else {
                localCache = localCacheDir.resolve(name);
            }
            logger = server.logger().addPrefix(name);
            uri = config.uri().toASCIIString();
            client =
                ClientBuilder.createClient(config, server.config().dnsConfig());
            closeChain.add(new GenericCloseableAdapter<>(client));
            updateInterval = config.updateInterval();
            tvm2Headers = config.tvm2Headers();
            tvm2HeadersSize = tvm2Headers.size();
            if (tvm2HeadersSize == 0) {
                ticketRenewalTask = null;
            } else {
                if (serviceContextRenewalTask == null) {
                    throw new ConfigException(
                        "TVM2 headers configured for external data '" + name
                        + "' but no TVM configured for service");
                }
                StringBuilder sb =
                    new StringBuilder(tvm2Headers.get(0).getValue());
                for (int i = 1; i < tvm2HeadersSize; ++i) {
                    sb.append(',');
                    sb.append(tvm2Headers.get(i).getValue());
                }
                try {
                    ticketRenewalTask =
                        new Tvm2TicketRenewalTask(
                            logger,
                            serviceContextRenewalTask,
                            new Tvm2ClientConfigBuilder()
                                .destinationClientId(new String(sb))
                                .renewalInterval(TVM_TICKETS_RENEWAL_INTERVAL)
                                .build());
                } catch (JsonException | URISyntaxException e) {
                    throw new IOException(e);
                }
                closeChain.add(new TimerTaskCloser(ticketRenewalTask));
                ticketRenewalTask.start();
            }
            holder.release();
        }
    }

    private String localCacheAttribute(final String name) throws IOException {
        Object attribute = Files.getAttribute(localCache, name);
        if (attribute == null) {
            return null;
        }
        ByteBuffer bb;
        if (attribute instanceof ByteBuffer) {
            bb = (ByteBuffer) attribute;
        } else {
            bb = ByteBuffer.wrap((byte[]) attribute);
        }
        return StandardCharsets.UTF_8.decode(bb).toString();
    }

    public synchronized void subscribe(final ExternalDataSubscriber subscriber)
        throws HttpException, IOException
    {
        int len = subscribers.length;
        ExternalDataSubscriber[] subscribers =
            Arrays.copyOf(this.subscribers, len + 1);
        subscribers[len] = subscriber;
        this.subscribers = subscribers;
        try {
            updateData();
        } catch (Throwable t) {
            logger.log(
                Level.WARNING,
                "Failed to fetch external data, "
                + "try fallback to local cache " + localCache,
                t);
            try {
                if (localCache != null && Files.exists(localCache)) {
                    String contentType =
                        localCacheAttribute(CONTENT_TYPE_ATTRIBUTE);
                    if (contentType == null) {
                        throw new IllegalStateException(
                            "Attibute " + CONTENT_TYPE_ATTRIBUTE
                            + " is not set for local cache " + localCache);
                    }
                    Header contentEncoding =
                        HeaderUtils.createHeader(
                            HttpHeaders.CONTENT_ENCODING,
                            localCacheAttribute(CONTENT_ENCODING_ATTRIBUTE));
                    updateData(
                        new ByteArrayProcessableWithContentType(
                            new ByteArrayProcessable(
                                Files.readAllBytes(localCache)),
                            ContentType.parse(contentType),
                            HeaderUtils.createHeader(
                                HttpHeaders.CONTENT_TYPE,
                                contentType),
                            contentEncoding));
                    lastUpdate =
                        Files.getLastModifiedTime(localCache).toMillis();
                    return;
                }
            } catch (Throwable t2) {
                logger.log(
                    Level.WARNING,
                    "Local cache fallback failed",
                    t2);
                t.addSuppressed(t2);
            }
            logger.warning("No suitable local cache found");
            lastUpdate = 0L;
            throw t;
        }
    }

    public String name() {
        return name;
    }

    public long updateInterval() {
        return updateInterval;
    }

    public long lastUpdate() {
        return lastUpdate;
    }

    private void updateData() throws HttpException, IOException {
        if (subscribers.length == 0) {
            logger.info("Empty subscribers. Skipping updateData()");
            return;
        }
        HttpGet get = new HttpGet(uri);
        for (int i = 0; i < tvm2HeadersSize; ++i) {
            Map.Entry<String, String> header = tvm2Headers.get(i);
            get.addHeader(
                header.getKey(),
                ticketRenewalTask.ticket(header.getValue()));
        }
        try (CloseableHttpResponse response = client.execute(get)) {
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(uri, response);
            }
            HttpEntity entity = response.getEntity();
            ContentType contentType = CharsetUtils.contentType(entity);
            Header contentEncoding = entity.getContentEncoding();
            Processable<byte[]> data = CharsetUtils.toDecodable(entity);
            updateData(
                new ByteArrayProcessableWithContentType(
                    data,
                    contentType,
                    entity.getContentType(),
                    contentEncoding));
            if (localCache != null) {
                Path tmp =
                    localCache.resolveSibling(
                        name + '_' + TimeSource.INSTANCE.currentTimeMillis());
                data.processWith(new PathWriterProcessor(tmp));
                Files.setAttribute(
                    tmp,
                    CONTENT_TYPE_ATTRIBUTE,
                    ByteBuffer.wrap(
                        StringUtils.getUtf8Bytes(contentType.toString())));
                String contentEncodingString;
                if (contentEncoding == null) {
                    contentEncodingString = "identity";
                } else {
                    contentEncodingString = contentEncoding.getValue();
                    if (contentEncodingString == null) {
                        contentEncodingString = "identity";
                    }
                }
                Files.setAttribute(
                    tmp,
                    CONTENT_ENCODING_ATTRIBUTE,
                    ByteBuffer.wrap(
                        StringUtils.getUtf8Bytes(contentEncodingString)));
                Files.move(
                    tmp,
                    localCache,
                    StandardCopyOption.REPLACE_EXISTING,
                    StandardCopyOption.ATOMIC_MOVE);
            }
        }
        if (localCache == null) {
            lastUpdate = TimeSource.INSTANCE.currentTimeMillis();
        } else {
            lastUpdate = Files.getLastModifiedTime(localCache).toMillis();
        }
    }

    private void updateData(final ByteArrayProcessableWithContentType data)
        throws HttpException, IOException
    {
        ExternalDataSubscriber[] subscribers = this.subscribers;
        for (ExternalDataSubscriber subscriber: subscribers) {
            try {
                subscriber.updateExternalData(data);
            } catch (Throwable t) {
                logger.log(
                    Level.WARNING,
                    "Failed to update external data for " + subscriber,
                    t);
                throw t;
            }
        }
    }

    @Override
    public void run() {
        try {
            updateData();
            logger.info("External data updated");
        } catch (Throwable t) {
            logger.log(
                Level.WARNING,
                "Failed to load external data from " + uri,
                t);
        }
    }
}

