package ru.yandex.http.util.server;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;

import com.sun.management.GarbageCollectionNotificationInfo;

import ru.yandex.logger.FilterLogger;

public final class GCMonitor {
    private static final CompoundLogger LOGGER = new CompoundLogger();
    private static MonitorImpl instance = null;
    private static Throwable createError = null;

    private final MonitorImpl impl;

    private GCMonitor(final Logger logger) {
        impl = implInstance(logger);
    }

    private GCMonitor() {
        impl = implInstance(null);
    }

    public static GCMonitor instance(final Logger logger) {
//        System.err.println("GCMonitor.instance: " + logger);
        return new GCMonitor(logger);
    }

    public static GCMonitor instance() {
        return new GCMonitor();
    }

    public long heapUsage() {
        return impl.heapUsage();
    }

    public long memoryLimit() {
        return impl.memoryLimit();
    }

    private static MonitorImpl implInstance(final Logger logger) {
        synchronized (LOGGER) {
            if (instance == null && createError == null) {
                try {
                    instance = createMonitor(LOGGER);
                } catch (Exception e) {
                    createError = e;
                    instance = new FakeGCMonitor();
                }
            }
            if (createError != null) {
                if (logger != null) {
                    logger.log(
                        Level.SEVERE,
                        "MemoryMonitor instantiate error",
                        createError);
                } else {
                    createError.printStackTrace();
                }
            }
            if (logger != null) {
                LOGGER.addLogger(logger);
            }
        }
        return instance;
    }

    private static MonitorImpl createMonitor(final Logger logger)
        throws Exception
    {
        return new GCListener(logger);
    }

    private static class FakeGCMonitor implements MonitorImpl {
        @Override
        public long heapUsage() {
            return -1;
        }

        @Override
        public long memoryLimit() {
            return -1;
        }
    }

    private static class GCListener
        implements MonitorImpl, NotificationListener
    {
        private static final String SHENANDOAH = "Shenandoah";
        private static final int MB = 1048576;
        private final Logger logger;
        private long totalPauseTime = 0;
        private long totalConcurrentTime = 0;
        private volatile long lastUsage;
        private long memoryLimit;

        GCListener(final Logger logger) {
            this.logger = logger;
            List<GarbageCollectorMXBean> gcbeans =
                java.lang.management.ManagementFactory
                    .getGarbageCollectorMXBeans();
            for (GarbageCollectorMXBean gcbean : gcbeans) {
//                System.err.println("eMonitoring: " + gcbean);
                logger.severe("Monitoring: " + gcbean);
                NotificationEmitter emitter = (NotificationEmitter) gcbean;
                emitter.addNotificationListener(this, null, gcbean);
            }
            MemoryMXBean mxBean = ManagementFactory.getMemoryMXBean();
            MemoryUsage usage = mxBean.getHeapMemoryUsage();
            lastUsage = usage.getUsed();
            memoryLimit = usage.getMax();
        }

        @Override
        public long memoryLimit() {
            return memoryLimit;
        }

        @Override
        public long heapUsage() {
            return lastUsage;
        }

        //CSOFF: MultipleStringLiterals
        @Override
        public void handleNotification(
            final Notification notification,
            final Object handback)
        {
            if (notification.getType().equals(
                GarbageCollectionNotificationInfo
                    .GARBAGE_COLLECTION_NOTIFICATION))
            {
                GarbageCollectionNotificationInfo info =
                    GarbageCollectionNotificationInfo.from(
                        (CompositeData) notification.getUserData());
                String gcName = info.getGcName();
                String action = info.getGcAction();
                long duration = info.getGcInfo().getDuration();
                logger.severe("GCMonitor: " + action + ": - "
                    + info.getGcInfo().getId()
                    + ' ' + info.getGcName()
                    + " (from " + info.getGcCause() + ") "
                    + duration + " milliseconds; start-end times "
                    + Long.toString(info.getGcInfo().getStartTime()) + '-'
                    + Long.toString(info.getGcInfo().getEndTime()));
                if (gcName.contains(SHENANDOAH)) {
                    parseShenandoah(info);
                } else if (gcName.contains("gencon")
                    || gcName.contains("scavenge")
                    || gcName.contains("global"))
                {
                    parseGencon(info);
                } else {
                    System.err.println(
                        "GCMonitor: unhandled garbage collector: " + gcName);
                    logger.severe("GCMonitor: unhandled garbage collector: "
                        + gcName + ", action: " + action
                        + ", cause: " + info.getGcCause());
//                        + notification.getUserData().toString());
//                        + gcName + ", usedData: "
//                        + notification.getUserData().toString());
                }
            }
        }

        private void parseGencon(
            final GarbageCollectionNotificationInfo info)
        {
            long duration = info.getGcInfo().getDuration();
            String action = info.getGcAction();
            if (!action.equals("end of minor GC")) {
//                totalConcurrentTime += duration;
                totalPauseTime += duration;
                Map<String, MemoryUsage> membefore =
                    info.getGcInfo().getMemoryUsageBeforeGc();
                Map<String, MemoryUsage> mem =
                    info.getGcInfo().getMemoryUsageAfterGc();
                long totalUsed = 0;
                long totalPrevUsed = 0;
                long max = 0;
                for (Map.Entry<String, MemoryUsage> entry : mem.entrySet()) {
                    String name = entry.getKey();
//                    if (!name.startsWith("nursery")
//                        && !name.startsWith("tenure"))
//                    {
//                        continue;
//                    }
                    MemoryUsage current = entry.getValue();
                    MemoryUsage before = membefore.get(name);
                    long used = current.getUsed();
                    logger.severe("GCMonitor: used: " + used + ", name: "
                        + name);
                    long prevUsed;
                    if (before != null) {
                        prevUsed = before.getUsed();
                    } else {
                        prevUsed = 0;
                    }
                    totalUsed += used;
                    totalPrevUsed += prevUsed;
                    max += current.getMax();
                }
                logger.severe("GCMonitor: used: " + (totalPrevUsed / MB)
                    + " -> " + (totalUsed / MB)
                    + ", free: " + (max - totalPrevUsed) / MB + " -> "
                    + (max - totalUsed) / MB);
                logger.severe("GCMonitor: totalPauseTime: "
                    + totalPauseTime + ", totalConcurrentTime: "
                    + totalConcurrentTime);
                lastUsage = totalUsed;
            } else {
                totalPauseTime += duration;
            }
        }

        private void parseShenandoah(
            final GarbageCollectionNotificationInfo info)
        {
            String action = info.getGcAction();
            long duration = info.getGcInfo().getDuration();
            if (action.equals("end of GC cycle")) {
                totalConcurrentTime += duration;
                Map<String, MemoryUsage> membefore =
                    info.getGcInfo().getMemoryUsageBeforeGc();
                Map<String, MemoryUsage> mem =
                    info.getGcInfo().getMemoryUsageAfterGc();
                long totalUsed = 0;
                long totalPrevUsed = 0;
                long max = 0;
                for (Map.Entry<String, MemoryUsage> entry : mem.entrySet()) {
                    String name = entry.getKey();
                    if (!name.equals(SHENANDOAH)) {
                        continue;
                    }
                    MemoryUsage current = entry.getValue();
                    MemoryUsage before = membefore.get(name);
                    long used = current.getUsed();
                    long prevUsed;
                    if (before != null) {
                        prevUsed = before.getUsed();
                    } else {
                        prevUsed = 0;
                    }
                    totalUsed += used;
                    totalPrevUsed += prevUsed;
                    max += current.getMax();
                }
                logger.severe("GCMonitor: used: " + (totalPrevUsed / MB)
                    + " -> " + (totalUsed / MB)
                    + ", free: " + (max - totalPrevUsed) / MB + " -> "
                    + (max - totalUsed) / MB);
                logger.severe("GCMonitor: totalPauseTime: "
                    + totalPauseTime + ", totalConcurrentTime: "
                    + totalConcurrentTime);
                lastUsage = totalUsed;
            } else {
                totalPauseTime += duration;
            }
        }
    }

    private interface MonitorImpl {
        long heapUsage();

        long memoryLimit();
    }

    private static class CompoundLogger extends FilterLogger {
        private final CopyOnWriteArrayList<Logger> loggers;

        CompoundLogger() {
            super(Logger.getAnonymousLogger());
            loggers = new CopyOnWriteArrayList<>();
        }

        public void addLogger(final Logger logger) {
//            System.err.println("GCMonitor.addLogger: " + logger
//                + " @" + this);
            loggers.add(logger);
        }

        @Override
        public void log(final LogRecord record) {
//            System.err.println("LOGGER.log");
            for (Logger logger: loggers) {
//                System.err.println("LOGGER.log.record: " + logger);
                logger.log(record);
            }
        }
    }
}
