package ru.yandex.http.util.server;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.channels.SocketChannel;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import javax.net.ssl.SSLContext;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.Pattern;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatersRegistrar;

public interface HttpServer<C extends ImmutableBaseServerConfig, T>
    extends
        ExternalDataProvider,
        GenericAutoCloseable<IOException>,
        Stater,
        StatersRegistrar
{
    // Context objects
    String SESSION_ID = "ru.yandex.http.util.session-id";
    String REQUEST_INFO = "ru.yandex.http.util.request-info";
    String LOGGER = "ru.yandex.http.util.session-logger";
    String TVM_SRC_ID = "ru.yandex.http.util.tvm-src-id";
    String TVM_USER_UID = "ru.yandex.http.util.tvm-user-uid";
    String SESSION_USER = "ru.yandex.http.util.session-user";
    String WARNING_MESSAGE = "ru.yandex.http.util.warning-message";
    String OVERRIDDEN_HANDLER = "ru.yandex.http.util.overridden-handler";

    // Status fields
    String LISTENING = "listening";
    String QUEUED_CONNECTIONS = "queued_connections";
    String ACTIVE_CONNECTIONS = "active_connections";
    String ACTIVE_WORKERS = "active_workers";
    String SPAWNED_WORKERS = "spawned_workers";

    String HTTP_PROTOCOL = "http";
    String LOCALHOST = "localhost";

    void start() throws IOException;

    InetSocketAddress address() throws IOException;

    InetSocketAddress httpAddress() throws IOException;

    SSLContext sslContext();

    boolean hasIBMDumper();

    default String scheme() {
        if (sslContext() == null) {
            return HTTP_PROTOCOL;
        } else {
            return "https";
        }
    }

    default int port() throws IOException {
        return address().getPort();
    }

    default int httpPort() throws IOException {
        return httpAddress().getPort();
    }

    default HttpHost host() throws IOException {
        return new HttpHost(LOCALHOST, port(), scheme());
    }

    default HttpHost httpHost() throws IOException {
        return new HttpHost(LOCALHOST, httpPort(), HTTP_PROTOCOL);
    }

    boolean pingEnabled(PrefixedLogger logger);

    void enablePing();

    void disablePing();

    Set<String> debugFlags();

    void addDebugFlag(String debugFlag);

    void removeDebugFlag(String debugFlag);

    void join() throws InterruptedException;

    void register(Pattern<RequestInfo> pattern, T handler);

    void register(Pattern<RequestInfo> pattern, T handler, String... methods);

    T register(Pattern<RequestInfo> pattern, T handler, String method);

    void unregister(Pattern<RequestInfo> pattern);

    void unregister(Pattern<RequestInfo> pattern, String... methods);

    T unregister(Pattern<RequestInfo> pattern, String method);

    void registerLoggerForLogrotate(Logger logger);

    void registerLoggerForLogrotate(HttpServer<?, ?> other);

    void rotateLogs(HttpContext context) throws HttpException;

    Map<String, Object> status(boolean verbose);

    List<Logger> loggers();

    C config();

    PrefixedLogger logger();

    void onIncomingConnection(SocketChannel channel);

    void onLimitedConnection(SocketChannel channel, String message);

    void discardChannel(SocketChannel channel);

    GolovanPanel golovanPanel(HttpRequest request) throws HttpException;

    GolovanPanel customGolovanPanel(HttpRequest request) throws HttpException;

    IniConfig alerts(HttpRequest request) throws HttpException;

    IniConfig customAlerts(HttpRequest request) throws HttpException;
}

