package ru.yandex.http.util.server;

import java.net.InetAddress;
import java.util.Objects;

import org.apache.http.Header;
import org.apache.http.HttpInetConnection;
import org.apache.http.HttpMessage;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpCoreContext;

import ru.yandex.logger.HashMapLookup;

public class HttpSessionInfo extends HashMapLookup {
    public static final String HEADER_PREFIX = "http_";
    public static final String SENT_HEADER_PREFIX = "sent_http_";

    private static final long serialVersionUID = 0L;

    private final transient HttpContext context;

    public HttpSessionInfo(final HttpContext context) {
        this.context = context;
    }

    private HttpInetConnection conn() {
        return (HttpInetConnection) context.getAttribute(
            HttpCoreContext.HTTP_CONNECTION);
    }

    private HttpRequest request() {
        return (HttpRequest) context.getAttribute(
            HttpCoreContext.HTTP_REQUEST);
    }

    private HttpResponse response() {
        return (HttpResponse) context.getAttribute(
            HttpCoreContext.HTTP_RESPONSE);
    }

    private String remoteAddr() {
        InetAddress address = conn().getRemoteAddress();
        if (address == null) {
            return null;
        } else {
            return address.getHostAddress();
        }
    }

    private String requestLine() {
        return request().getRequestLine().toString();
    }

    private String requestUri() {
        return request().getRequestLine().getUri();
    }

    private String requestId() {
        return (String) context.getAttribute(HttpServer.SESSION_ID);
    }

    private String user() {
        return (String) context.getAttribute(HttpServer.SESSION_USER);
    }

    private static String header(
        final String name,
        final int prefixLength,
        final HttpMessage message)
    {
        String headerName = name.substring(prefixLength).replace('_', '-');
        Header header = message.getFirstHeader(headerName);
        if (header == null) {
            return null;
        } else {
            return header.getValue();
        }
    }

    @Override
    public String get(final Object name) {
        String value = Objects.toString(super.get(name), null);
        if (value == null) {
            String nameString = name.toString();
            if (nameString.startsWith(HEADER_PREFIX)) {
                value = header(nameString, HEADER_PREFIX.length(), request());
            } else if (nameString.startsWith(SENT_HEADER_PREFIX)) {
                HttpResponse response = response();
                if (response != null) {
                    value = header(
                        nameString,
                        SENT_HEADER_PREFIX.length(),
                        response);
                }
            }
        }
        return value;
    }

    @Override
    public Object lookup(final String name) {
        switch (name) {
            case "remote_addr":
                return remoteAddr();
            case "request":
                return requestLine();
            case "uri":
                return requestUri();
            case "session_id":
                return requestId();
            case "user":
                return user();
            default:
                return get(name);
        }
    }
}

