package ru.yandex.http.util.server;

import java.io.IOException;
import java.lang.reflect.Method;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.PosixFilePermission;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.util.filesystem.PathUtils;

public class IBMDumper {
    private static final String FOLDER = "folder=";

    private final Method javaDump;
    private final Method systemDump;
    private final Method heapDump;

    public IBMDumper() throws ReflectiveOperationException {
        Class<?> dump =
            ClassLoader.getSystemClassLoader().loadClass("com.ibm.jvm.Dump");
        javaDump = dump.getMethod("JavaDump");
        systemDump = dump.getMethod("SystemDump");
        heapDump = dump.getMethod("HeapDump");
    }

    public void javaDump() throws ReflectiveOperationException {
        javaDump.invoke(null);
    }

    public void systemDump() throws ReflectiveOperationException {
        systemDump.invoke(null);
    }

    public void heapDump() throws ReflectiveOperationException {
        heapDump.invoke(null);
    }

    public void process(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        try {
            String uri = request.getRequestLine().getUri();
            int index = uri.indexOf('?');
            String path;
            Path folder;
            if (index == -1) {
                path = uri;
                folder = null;
            } else {
                path = uri.substring(0, index);
                String rest = uri.substring(index + 1);
                if (rest.startsWith(FOLDER)) {
                    folder = FileSystems.getDefault()
                        .getPath(rest.substring(FOLDER.length()));
                } else {
                    folder = null;
                }
            }
            Set<Path> oldFiles;
            if (folder == null) {
                oldFiles = null;
            } else {
                oldFiles =
                    PathUtils.listFiles(folder, LinkOption.NOFOLLOW_LINKS);
            }
            switch (path) {
                case "/javadump":
                    javaDump();
                    break;
                case "/systemdump":
                    systemDump();
                    break;
                case "/heapdump":
                    heapDump();
                    break;
                default:
                    throw new BadRequestException("Uri not supported: "
                        + request.getRequestLine().getUri());
            }
            if (folder != null) {
                Set<Path> newFiles =
                    PathUtils.listFiles(folder, LinkOption.NOFOLLOW_LINKS);
                newFiles.removeAll(oldFiles);
                Logger logger =
                    (Logger) context.getAttribute(HttpServer.LOGGER);
                for (Path file: newFiles) {
                    Set<PosixFilePermission> perms =
                        Files.getPosixFilePermissions(file);
                    if (!perms.contains(PosixFilePermission.OTHERS_READ)) {
                        perms.add(PosixFilePermission.OTHERS_READ);
                        logger.info("Setting " + perms + " for " + file);
                        Files.setPosixFilePermissions(file, perms);
                    }
                }
            }
        } catch (IOException e) {
            throw new ServiceUnavailableException(e);
        } catch (ReflectiveOperationException e) {
            throw new NotImplementedException(e);
        }
    }
}

