package ru.yandex.http.util.server;

import ru.yandex.client.tvm2.ImmutableTvm2ServiceConfig;
import ru.yandex.http.config.ImmutableDnsConfig;
import ru.yandex.logger.ImmutableLoggersConfig;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.ImmutableStatersConfig;

public class ImmutableBaseServerConfig
    extends ImmutableHttpServerConfig
    implements BaseServerConfig
{
    private final ImmutableDnsConfig dnsConfig;
    private final ImmutableTvm2ServiceConfig tvm2ServiceConfig;
    private final ImmutableLoggersConfig loggers;
    private final ImmutableStatersConfig staters;
    private final ImmutableAuthsConfig auths;
    private final ImmutableLimitersConfig limiters;
    private final ImmutableGolovanPanelConfig golovanPanel;
    private final ImmutableAutoRegisterRequestStaterConfig autoRegisterRequestStater;

    public ImmutableBaseServerConfig(final BaseServerConfig config)
        throws ConfigException
    {
        this(
            config,
            new ImmutableDnsConfig(config.dnsConfig()),
            ImmutableTvm2ServiceConfig.create(config.tvm2ServiceConfig()),
            new ImmutableLoggersConfig(config.loggers()),
            new ImmutableStatersConfig(config.staters()),
            new ImmutableAuthsConfig(config.auths()),
            new ImmutableLimitersConfig(config.limiters()),
            ImmutableGolovanPanelConfig.create(config.golovanPanel()),
            new ImmutableAutoRegisterRequestStaterConfig(
                config.autoRegisterRequestStater()));
    }

    public ImmutableBaseServerConfig(
        final HttpServerConfig config,
        final ImmutableDnsConfig dnsConfig,
        final ImmutableTvm2ServiceConfig tvm2ServiceConfig,
        final ImmutableLoggersConfig loggers,
        final ImmutableStatersConfig staters,
        final ImmutableAuthsConfig auths,
        final ImmutableLimitersConfig limiters,
        final ImmutableGolovanPanelConfig golovanPanel,
        final ImmutableAutoRegisterRequestStaterConfig
            autoRegisterRequestStater)
        throws ConfigException
    {
        super(config);
        this.dnsConfig = dnsConfig;
        this.tvm2ServiceConfig = tvm2ServiceConfig;
        this.loggers = loggers;
        this.staters = staters;
        this.auths = auths;
        this.limiters = limiters;
        this.golovanPanel = golovanPanel;
        this.autoRegisterRequestStater = autoRegisterRequestStater;

        ImmutableAuthConfig asteriskAuth = auths.auths().asterisk();
        boolean hasAuths =
            !auths.auths().asteriskOnly()
            || !asteriskAuth.disabled()
            || asteriskAuth.userTicketPresence() != UserTicketPresence.IGNORED;
        if (hasAuths) {
            forceTvm2ServiceConfig();
        }
    }

    protected void forceTvm2ServiceConfig() throws ConfigException {
        if (tvm2ServiceConfig == null) {
            throw new ConfigException("tvm2 service is not configured");
        }
    }

    @Override
    public ImmutableDnsConfig dnsConfig() {
        return dnsConfig;
    }

    @Override
    public ImmutableTvm2ServiceConfig tvm2ServiceConfig() {
        return tvm2ServiceConfig;
    }

    @Override
    public ImmutableLoggersConfig loggers() {
        return loggers;
    }

    @Override
    public ImmutableStatersConfig staters() {
        return staters;
    }

    @Override
    public ImmutableAuthsConfig auths() {
        return auths;
    }

    @Override
    public ImmutableLimitersConfig limiters() {
        return limiters;
    }

    @Override
    public ImmutableGolovanPanelConfig golovanPanel() {
        return golovanPanel;
    }

    public long metricsTimeFrame() {
        return staters.staters().asterisk().metricsTimeFrame();
    }

    @Override
    public ImmutableAutoRegisterRequestStaterConfig autoRegisterRequestStater() {
        return autoRegisterRequestStater;
    }
}

