package ru.yandex.http.util.server;

import java.nio.file.FileStore;
import java.nio.file.Path;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.http.Header;

import ru.yandex.collection.CollectionCompactor;
import ru.yandex.collection.LongPair;
import ru.yandex.http.config.ExternalDataConfig;
import ru.yandex.http.config.ImmutableExternalDataConfig;
import ru.yandex.http.config.ImmutableHttpConnectionConfig;
import ru.yandex.http.config.ImmutableServerHttpsConfig;
import ru.yandex.http.config.ImmutableURICheckConfig;
import ru.yandex.http.config.ServerHttpsConfig;
import ru.yandex.http.config.URICheckConfig;
import ru.yandex.http.util.request.function.NullFunction;
import ru.yandex.parser.config.ConfigException;

public class ImmutableHttpServerConfig
    extends ImmutableHttpConnectionConfig
    implements HttpServerConfig
{
    private final String name;
    private final String origin;
    private final int port;
    private final int workers;
    private final int connections;
    private final int maxGarbageConnections;
    private final int minConnectionsPerHost;
    private final int maxConnectionsPerHost;
    private final boolean rejectConnectionsOverLimit;
    private final int timeout;
    private final long connectionCloseCheckInterval;
    private final int timerResolution;
    private final int backlog;
    private final int linger;
    private final boolean gzip;
    private final boolean lazyBind;
    private final boolean pingEnabledOnStartup;
    private final Set<String> hiddenHeaders;
    private final List<Header> staticHeaders;
    private final List<LongPair<String>> staticStats;
    private final Map<String, Set<String>> statsAliases;
    private final String statsPrefix;
    private final boolean keepUnprefixedStats;
    private final Map<String, FileStore> freeSpaceSignals;
    private final Map<String, ImmutableFilesStaterConfig> filesStaters;
    private final ImmutableServerHttpsConfig httpsConfig;
    private final boolean cpuStater;
    private final boolean memoryStater;
    private final boolean gcStater;
    private final boolean heapStater;
    private final boolean instanceAliveStater;
    private final Limiter connectionsLimiter;
    private final Set<String> debugFlags;
    private final Set<Integer> keepAliveStatuses;
    private final Set<String> defaultHttpMethods;
    private final Map<String, ImmutableURICheckConfig> httpChecks;
    private final Path localCacheDir;
    private final Map<String, ImmutableExternalDataConfig> externalData;

    public ImmutableHttpServerConfig(final HttpServerConfig config)
        throws ConfigException
    {
        super(config);
        name = config.name();
        origin = config.origin();
        port = config.port();
        workers = config.workers();
        connections = config.connections();
        maxGarbageConnections = config.maxGarbageConnections();
        minConnectionsPerHost = config.minConnectionsPerHost();
        maxConnectionsPerHost = config.maxConnectionsPerHost();
        rejectConnectionsOverLimit = config.rejectConnectionsOverLimit();
        timeout = config.timeout();
        connectionCloseCheckInterval = config.connectionCloseCheckInterval();
        timerResolution = config.timerResolution();
        backlog = config.backlog();
        linger = config.linger();
        gzip = config.gzip();
        lazyBind = config.lazyBind();
        pingEnabledOnStartup = config.pingEnabledOnStartup();
        cpuStater = config.cpuStater();
        memoryStater = config.memoryStater();
        gcStater = config.gcStater();
        heapStater = config.heapStater();
        instanceAliveStater = config.instanceAliveStater();
        hiddenHeaders = new TreeSet<>(String.CASE_INSENSITIVE_ORDER);
        hiddenHeaders.addAll(config.hiddenHeaders());
        staticHeaders = config.staticHeaders();
        staticStats = config.staticStats();

        Map<String, Set<String>> statsAliases =
            new HashMap<>(config.statsAliases());
        Iterator<Map.Entry<String, Set<String>>> iter =
            statsAliases.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, Set<String>> entry = iter.next();
            Set<String> aliases = entry.getValue();
            switch (aliases.size()) {
                case 0:
                    iter.remove();
                    break;
                case 1:
                    entry.setValue(
                        Collections.singleton(aliases.iterator().next()));
                    break;
                default:
                    entry.setValue(new HashSet<>(aliases));
                    break;
            }
        }
        this.statsAliases = CollectionCompactor.compact(statsAliases);
        statsPrefix = config.statsPrefix();
        keepUnprefixedStats = config.keepUnprefixedStats();

        freeSpaceSignals = CollectionCompactor.compact(
            new LinkedHashMap<>(config.freeSpaceSignals()));

        Map<String, ? extends FilesStaterConfig> filesStaters =
            config.filesStaters();
        Map<String, ImmutableFilesStaterConfig> filesStatersCopy =
            new LinkedHashMap<>(filesStaters.size() << 1);
        for (Map.Entry<String, ? extends FilesStaterConfig> entry
            : filesStaters.entrySet())
        {
            String name = entry.getKey();
            filesStatersCopy.put(
                name,
                HttpServerConfig.filesStaterParser(name)
                    .validate(entry.getValue()));
        }
        this.filesStaters = CollectionCompactor.compact(filesStatersCopy);

        ServerHttpsConfig httpsConfig = config.httpsConfig();
        if (httpsConfig == null) {
            this.httpsConfig = null;
        } else {
            this.httpsConfig = new ImmutableServerHttpsConfig(httpsConfig);
        }

        debugFlags =
            CollectionCompactor.compact(new HashSet<>(config.debugFlags()));
        keepAliveStatuses =
            CollectionCompactor.compact(
                new HashSet<>(config.keepAliveStatuses()));
        defaultHttpMethods = Collections.unmodifiableSet(
            new LinkedHashSet<>(
                config.defaultHttpMethods()));

        Map<String, ? extends URICheckConfig> httpChecks = config.httpChecks();
        Map<String, ImmutableURICheckConfig> httpChecksCopy =
            new LinkedHashMap<>(httpChecks.size() << 1);
        for (Map.Entry<String, ? extends URICheckConfig> entry
            : httpChecks.entrySet())
        {
            String name = entry.getKey();
            httpChecksCopy.put(
                name,
                HttpServerConfig.httpCheckParser(name)
                    .validate(entry.getValue()));
        }
        this.httpChecks = CollectionCompactor.compact(httpChecksCopy);

        localCacheDir = config.localCacheDir();

        Map<String, ? extends ExternalDataConfig> externalData =
            config.externalData();
        Map<String, ImmutableExternalDataConfig> externalDataCopy =
            new LinkedHashMap<>(externalData.size() << 1);
        for (Map.Entry<String, ? extends ExternalDataConfig> entry
            : externalData.entrySet())
        {
            String name = entry.getKey();
            externalDataCopy.put(
                name,
                HttpServerConfig.externalDataConfigParser(name)
                    .validate(entry.getValue()));
        }
        this.externalData = CollectionCompactor.compact(externalDataCopy);

        if (port < 0) {
            throw new ConfigException("Port is not configured");
        }

        if (connections < 0) {
            throw new ConfigException("Connections count is not configured");
        }

        if (maxGarbageConnections <= 0) {
            throw new ConfigException(
                "Max garbase connections count is not configured");
        }

        if (minConnectionsPerHost < 0) {
            throw new ConfigException(
                "min-connections-per-host must be non-negative value");
        }

        if (maxConnectionsPerHost <= minConnectionsPerHost) {
            throw new ConfigException(
                "max-connections-per-host must be greater "
                + "than min-connections-per-host");
        }

        LimiterConfigBuilder limiterConfig = new LimiterConfigBuilder();
        limiterConfig.staterPrefix("connections");
        limiterConfig.concurrency(connections);
        boolean hasPerHostConcurrency = false;
        if (maxConnectionsPerHost != Integer.MAX_VALUE) {
            hasPerHostConcurrency = true;
            limiterConfig.perKeyConcurrency(maxConnectionsPerHost);
        }
        if (minConnectionsPerHost > 0) {
            hasPerHostConcurrency = true;
            limiterConfig.perKeyMinimalConcurrency(minConnectionsPerHost);
        }
        if (hasPerHostConcurrency) {
            limiterConfig.key(NullFunction.instance());
        }
        connectionsLimiter = limiterConfig.build().build();
    }

    @Override
    public String name() {
        return name;
    }

    @Override
    public String origin() {
        return origin;
    }

    @Override
    public int port() {
        return port;
    }

    @Override
    public int workers() {
        return workers;
    }

    @Override
    public int connections() {
        return connections;
    }

    @Override
    public int maxGarbageConnections() {
        return maxGarbageConnections;
    }

    @Override
    public int minConnectionsPerHost() {
        return minConnectionsPerHost;
    }

    @Override
    public int maxConnectionsPerHost() {
        return maxConnectionsPerHost;
    }

    @Override
    public boolean rejectConnectionsOverLimit() {
        return rejectConnectionsOverLimit;
    }

    @Override
    public int timeout() {
        return timeout;
    }

    @Override
    public long connectionCloseCheckInterval() {
        return connectionCloseCheckInterval;
    }

    @Override
    public int timerResolution() {
        return timerResolution;
    }

    @Override
    public int backlog() {
        return backlog;
    }

    @Override
    public int linger() {
        return linger;
    }

    @Override
    public boolean gzip() {
        return gzip;
    }

    @Override
    public boolean lazyBind() {
        return lazyBind;
    }

    @Override
    public boolean pingEnabledOnStartup() {
        return pingEnabledOnStartup;
    }

    @Override
    public boolean cpuStater() {
        return cpuStater;
    }

    @Override
    public boolean memoryStater() {
        return memoryStater;
    }

    @Override
    public boolean gcStater() {
        return gcStater;
    }

    @Override
    public boolean heapStater() {
        return heapStater;
    }

    @Override
    public boolean instanceAliveStater() {
        return instanceAliveStater;
    }

    @Override
    public Set<String> hiddenHeaders() {
        return hiddenHeaders;
    }

    @Override
    public List<Header> staticHeaders() {
        return staticHeaders;
    }

    @Override
    public List<LongPair<String>> staticStats() {
        return staticStats;
    }

    @Override
    public Map<String, Set<String>> statsAliases() {
        return statsAliases;
    }

    @Override
    public String statsPrefix() {
        return statsPrefix;
    }

    @Override
    public boolean keepUnprefixedStats() {
        return keepUnprefixedStats;
    }

    @Override
    public Map<String, FileStore> freeSpaceSignals() {
        return freeSpaceSignals;
    }

    @Override
    public Map<String, ImmutableFilesStaterConfig> filesStaters() {
        return filesStaters;
    }

    @Override
    public ImmutableServerHttpsConfig httpsConfig() {
        return httpsConfig;
    }

    @Override
    public Set<String> debugFlags() {
        return debugFlags;
    }

    @Override
    public Set<Integer> keepAliveStatuses() {
        return keepAliveStatuses;
    }

    @Override
    public Set<String> defaultHttpMethods() {
        return defaultHttpMethods;
    }

    @Override
    public Map<String, ImmutableURICheckConfig> httpChecks() {
        return httpChecks;
    }

    @Override
    public Path localCacheDir() {
        return localCacheDir;
    }

    @Override
    public Map<String, ImmutableExternalDataConfig> externalData() {
        return externalData;
    }

    public Limiter connectionsLimiter() {
        return connectionsLimiter;
    }
}

