package ru.yandex.http.util.server;

import ru.yandex.http.util.request.function.RequestFunction;
import ru.yandex.parser.config.ConfigException;

public class ImmutableLimiterConfig implements LimiterConfig {
    private static final String WITH_NO_KEY_DEFINED = ") with no key defined";
    private static final String MINIMAL_CONCURRENCY_IS_SET =
        "per-key-minimal-concurrency is set (";

    private final int concurrency;
    private final int perKeyConcurrency;
    private final int perKeyMinimalConcurrency;
    private final RequestFunction<LimiterFunctionContext> key;
    private final long memoryLimit;
    private final int errorStatusCode;
    private final String staterPrefix;
    private final boolean bypassLoopback;

    public ImmutableLimiterConfig(final LimiterConfig config)
        throws ConfigException
    {
        concurrency = CONCURRENCY.validate(config.concurrency());
        perKeyConcurrency =
            PER_KEY_CONCURRENCY.validate(config.perKeyConcurrency());
        perKeyMinimalConcurrency =
            PER_KEY_CONCURRENCY.validate(config.perKeyMinimalConcurrency());
        key = KEY.validate(config.key());
        memoryLimit = MEMORY_LIMIT.validate(config.memoryLimit());
        errorStatusCode =
            ERROR_STATUS_CODE.validate(config.errorStatusCode());
        staterPrefix = STATER_PREFIX.validate(config.staterPrefix());
        bypassLoopback = BYPASS_LOOPBACK.validate(config.bypassLoopback());

        if (key == null) {
            if (perKeyConcurrency != -1) {
                throw new ConfigException(
                    "per-key-concurrency is set ("
                    + perKeyConcurrency + WITH_NO_KEY_DEFINED);
            }
            if (perKeyMinimalConcurrency != -1) {
                throw new ConfigException(
                    MINIMAL_CONCURRENCY_IS_SET
                    + perKeyMinimalConcurrency + WITH_NO_KEY_DEFINED);
            }
        } else {
            if (perKeyMinimalConcurrency == -1) {
                if (perKeyConcurrency == -1) {
                    throw new ConfigException(
                        "limiter key is set (" + key
                        + ") with neither per-key-concurrency nor "
                        + "per-key-minimal-concurrency limit defined");
                }
            } else if (concurrency == -1 && memoryLimit == -1L) {
                throw new ConfigException(
                    MINIMAL_CONCURRENCY_IS_SET + perKeyMinimalConcurrency
                    + ") but no all requests wide limits is set");
            }
        }
    }

    @Override
    public int concurrency() {
        return concurrency;
    }

    @Override
    public int perKeyConcurrency() {
        return perKeyConcurrency;
    }

    @Override
    public int perKeyMinimalConcurrency() {
        return perKeyMinimalConcurrency;
    }

    @Override
    public RequestFunction<LimiterFunctionContext> key() {
        return key;
    }

    @Override
    public long memoryLimit() {
        return memoryLimit;
    }

    @Override
    public int errorStatusCode() {
        return errorStatusCode;
    }

    @Override
    public String staterPrefix() {
        return staterPrefix;
    }

    public Limiter build() {
        return new Limiter(this);
    }

    @Override
    public boolean bypassLoopback() {
        return bypassLoopback;
    }

    @Override
    public String toString() {
        return "ImmutableLimiterConfig{" +
            "concurrency=" + concurrency +
            ", perKeyConcurrency=" + perKeyConcurrency +
            ", perKeyMinimalConcurrency=" + perKeyMinimalConcurrency +
            ", key=" + key +
            ", memoryLimit=" + memoryLimit +
            ", errorStatusCode=" + errorStatusCode +
            ", bypassLoopback=" + bypassLoopback +
            '}';
    }
}
