package ru.yandex.http.util.server;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.HttpRequest;

import ru.yandex.concurrent.LockStorage;
import ru.yandex.function.FinalizingGenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.request.function.RequestFunction;
import ru.yandex.http.util.request.function.RequestFunctionValue;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanPanelConfig;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class Limiter implements Serializable, Stater {
    private static final long serialVersionUID = 0L;

    private final AtomicInteger concurrentRequests;
    private final AtomicLong concurrentRequestsSize;
    private final boolean bypassLoopback;
    private final int limit;
    private final long memoryLimit;
    private final int perKeyLimit;
    private final int perKeyMinimalLimit;
    private final LockStorage<RequestFunctionValue, AtomicInteger> keyRequests;
    private final RequestFunction<LimiterFunctionContext> key;
    private final boolean enabled;
    private final int errorStatusCode;
    private final String errorPhrase;
    private final String prefix;
    private final String currentRequestsSignalAxxx;
    private final String currentRequestsSignalAmmm;
    private final String maxRequestsSignalAxxx;
    private final String maxRequestsSignalAmmm;
    private final String currentMemorySignalAxxx;
    private final String currentMemorySignalAmmm;
    private final String maxMemorySignalAxxx;
    private final String maxMemorySignalAmmm;

    public Limiter(final ImmutableLimiterConfig config) {
        this(config, null);
    }

    public Limiter(final ImmutableLimiterConfig config, final Limiter other) {
        if (other == null) {
            concurrentRequests = new AtomicInteger(0);
            concurrentRequestsSize = new AtomicLong(0);
        } else {
            concurrentRequests = other.concurrentRequests;
            concurrentRequestsSize = other.concurrentRequestsSize;
        }

        bypassLoopback = config.bypassLoopback();
        int limit = config.concurrency();
        if (limit == -1) {
            this.limit = Integer.MAX_VALUE;
        } else {
            this.limit = limit;
        }
        long memoryLimit = config.memoryLimit();
        if (memoryLimit == -1L) {
            this.memoryLimit = Long.MAX_VALUE;
        } else {
            this.memoryLimit = memoryLimit;
        }
        int perKeyLimit = config.perKeyConcurrency();
        this.perKeyMinimalLimit = config.perKeyMinimalConcurrency();
        if (perKeyLimit == -1 && perKeyMinimalLimit == -1) {
            this.perKeyLimit = Integer.MAX_VALUE;
            this.keyRequests = null;
            this.key = null;
        } else {
            if (perKeyLimit == -1) {
                this.perKeyLimit = Integer.MAX_VALUE;
            } else {
                this.perKeyLimit = perKeyLimit;
            }
            if (other == null) {
                this.keyRequests = new LockStorage<>();
            } else {
                this.keyRequests = other.keyRequests;
            }

            this.key = config.key();
        }
        enabled = limit != Integer.MAX_VALUE
            || memoryLimit != Long.MAX_VALUE
            || perKeyLimit != Integer.MAX_VALUE;
        errorStatusCode = config.errorStatusCode();
        errorPhrase = "Concurrent connections limit is " + limit;
        prefix = config.staterPrefix();
        if (prefix == null) {
            currentRequestsSignalAxxx = null;
            currentRequestsSignalAmmm = null;
            maxRequestsSignalAxxx = null;
            maxRequestsSignalAmmm = null;
            currentMemorySignalAxxx = null;
            currentMemorySignalAmmm = null;
            maxMemorySignalAxxx = null;
            maxMemorySignalAmmm = null;
        } else {
            currentRequestsSignalAxxx = prefix + "-current-requests_axxx";
            currentRequestsSignalAmmm = prefix + "-current-requests_ammm";
            maxRequestsSignalAxxx = prefix + "-max-requests_axxx";
            maxRequestsSignalAmmm = prefix + "-max-requests_ammm";
            currentMemorySignalAxxx = prefix + "-current-memory_axxx";
            currentMemorySignalAmmm = prefix + "-current-memory_ammm";
            maxMemorySignalAxxx = prefix + "-max-memory_axxx";
            maxMemorySignalAmmm = prefix + "-max-memory_ammm";
        }
    }

    public int errorStatusCode() {
        return errorStatusCode;
    }

    public boolean enabled() {
        return enabled;
    }

    public boolean bypassLoopback() {
        return bypassLoopback;
    }

    public boolean staterEnabled() {
        return currentRequestsSignalAxxx != null;
    }

    public boolean perKeyLimitsEnabled() {
        return key != null;
    }

    public RequestFunctionValue key(final HttpRequest request)
        throws ExecutionException
    {
        if (key == null) {
            return null;
        } else {
            return key.value(new LimiterFunctionContext(request));
        }
    }

    private String check(
        final int requests,
        final int currentKeyRequests,
        final long memory,
        final long contentLength,
        final RequestFunctionValue key)
    {
        String message;
        if (currentKeyRequests <= perKeyMinimalLimit) {
            message = null;
        } else if (requests > limit) {
            message = errorPhrase;
        } else if (memory > memoryLimit && requests > 1) {
            message =
                "Concurrent requests size is " + memoryLimit
                + ", current memory usage is " + memory
                + ", rejecting request of size " + contentLength;
        } else if (currentKeyRequests > perKeyLimit) {
            message =
                "Concurrent requests for key " + this.key
                + " == " + key + " has been reached: "
                + (currentKeyRequests - 1);
        } else {
            message = null;
        }
        return message;
    }

    public LimiterResult acquire(
        final long contentLength,
        final RequestFunctionValue key)
    {
        AtomicInteger keyRequestsCounter;
        int currentKeyRequests;
        if (key == null) {
            keyRequestsCounter = null;
            currentKeyRequests = Integer.MAX_VALUE;
        } else {
            keyRequestsCounter = keyRequests.acquire(key, new AtomicInteger());
            currentKeyRequests = keyRequestsCounter.get() + 1;
        }
        int requests = concurrentRequests.get() + 1;
        long memory;
        if (contentLength > 0) {
            memory = concurrentRequestsSize.get() + contentLength;
        } else {
            // pass requests without content length, like /stat and /ping
            memory = 0;
        }
        String message =
            check(requests, currentKeyRequests, memory, contentLength, key);
        if (message != null) {
            if (key != null) {
                keyRequests.release(key);
            }
            return new LimiterResult(message, null);
        }
        FinalizingGenericAutoCloseableChain<RuntimeException> chain =
            new FinalizingGenericAutoCloseableChain<>();
        requests = concurrentRequests.incrementAndGet();
        chain.add(new RequestsReleaser(concurrentRequests));
        if (contentLength > 0) {
            memory = concurrentRequestsSize.addAndGet(contentLength);
            chain.add(
                new MemoryReleaser(concurrentRequestsSize, contentLength));
        }
        if (key != null) {
            currentKeyRequests = keyRequestsCounter.incrementAndGet();
            chain.add(
                new KeyRequestsReleaser(keyRequests, key, keyRequestsCounter));
        }
        message =
            check(requests, currentKeyRequests, memory, contentLength, key);
        // Request rejected, rollback counters
        if (message != null) {
            // Will make isEmpty() == true
            chain.close();
        }
        if (chain.isEmpty()) {
            chain = null;
        }
        return new LimiterResult(message, chain);
    }

    public int concurrentRequests() {
        return concurrentRequests.get();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        int concurrentRequests = this.concurrentRequests.get();
        statsConsumer.stat(currentRequestsSignalAxxx, concurrentRequests);
        statsConsumer.stat(currentRequestsSignalAmmm, concurrentRequests);
        if (limit < Integer.MAX_VALUE) {
            statsConsumer.stat(maxRequestsSignalAxxx, limit);
            statsConsumer.stat(maxRequestsSignalAmmm, limit);
        }
        long concurrentRequestsSize = this.concurrentRequestsSize.get();
        statsConsumer.stat(currentMemorySignalAxxx, concurrentRequestsSize);
        statsConsumer.stat(currentMemorySignalAmmm, concurrentRequestsSize);
        if (memoryLimit < Long.MAX_VALUE) {
            statsConsumer.stat(maxMemorySignalAxxx, memoryLimit);
            statsConsumer.stat(maxMemorySignalAmmm, memoryLimit);
        }
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix)
    {
        String prefix = StringUtils.removeSuffix(this.prefix, '-');
        String chartsPrefix = statsPrefix + prefix;
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);
        ImmutableGolovanPanelConfig config = panel.config();

        GolovanChart requestsChart = new GolovanChart(
            "-concurrent-requests",
            " concurrent requests",
            false,
            false,
            0d);
        requestsChart.addSignal(
            new GolovanSignal(
                chartsPrefix + "-current-requests_axxx",
                config.tag(),
                "max",
                null,
                0,
                false));
        requestsChart.addSignal(
            new GolovanSignal(
                "div(" + chartsPrefix + "-current-requests_ammm,"
                + statsPrefix + "instance-alive_ammm)",
                config.tag(),
                "average",
                null,
                1,
                false));
        if (limit < Integer.MAX_VALUE) {
            requestsChart.addSignal(
                new GolovanSignal(
                    chartsPrefix + "-max-requests_axxx",
                    config.tag(),
                    "limit",
                    "#ff0000",
                    0,
                    false));
        }
        group.addChart(requestsChart);

        GolovanChart requestsSizeChart = new GolovanChart(
            "-concurrent-requests-size",
            " concurrent requests size (bytes)",
            false,
            false,
            0d);
        requestsSizeChart.addSignal(
            new GolovanSignal(
                chartsPrefix + "-current-memory_axxx",
                config.tag(),
                "max",
                null,
                0,
                false));
        requestsSizeChart.addSignal(
            new GolovanSignal(
                "div(" + chartsPrefix + "-current-memory_ammm,"
                + statsPrefix + "instance-alive_ammm)",
                config.tag(),
                "average",
                null,
                1,
                false));
        if (memoryLimit < Long.MAX_VALUE) {
            requestsSizeChart.addSignal(
                new GolovanSignal(
                    chartsPrefix + "-max-memory_axxx",
                    config.tag(),
                    "limit",
                    "#ff0000",
                    0,
                    false));
        }
        group.addChart(requestsSizeChart);

        panel.addCharts(
            GolovanPanelConfig.CATEGORY_LIMITER,
            null,
            group);
    }

    @Override
    public String toString() {
        return "Limiter{" +
            "currentState=" + concurrentRequests.get() +
            ", limit=" + limit +
            ", memoryLimit=" + memoryLimit +
            ", perKeyLimit=" + perKeyLimit +
            ", perKeyMinimalLimit=" + perKeyMinimalLimit +
            ", errorStatusCode=" + errorStatusCode +
            '}';
    }

    private void writeObject(final ObjectOutputStream stream)
        throws IOException
    {
        stream.defaultWriteObject();
    }

    private void readObject(final ObjectInputStream stream)
        throws IOException, ClassNotFoundException
    {
        stream.defaultReadObject();
    }

    private static class RequestsReleaser
        implements GenericAutoCloseable<RuntimeException>
    {
        private final AtomicInteger requestsCounter;

        RequestsReleaser(final AtomicInteger requestsCounter) {
            this.requestsCounter = requestsCounter;
        }

        @Override
        public void close() {
            requestsCounter.decrementAndGet();
        }
    }

    private static class MemoryReleaser
        implements GenericAutoCloseable<RuntimeException>
    {
        private final AtomicLong memoryCounter;
        private final long memory;

        MemoryReleaser(final AtomicLong memoryCounter, final long memory) {
            this.memoryCounter = memoryCounter;
            this.memory = memory;
        }

        @Override
        public void close() {
            memoryCounter.addAndGet(-memory);
        }
    }

    private static class KeyRequestsReleaser
        implements GenericAutoCloseable<RuntimeException>
    {
        private final LockStorage<?, ?> keyRequestsStorage;
        private final RequestFunctionValue key;
        private final AtomicInteger keyRequestsCounter;

        KeyRequestsReleaser(
            final LockStorage<?, ?> keyRequestsStorage,
            final RequestFunctionValue key,
            final AtomicInteger keyRequestsCounter)
        {
            this.keyRequestsStorage = keyRequestsStorage;
            this.key = key;
            this.keyRequestsCounter = keyRequestsCounter;
        }

        @Override
        public void close() {
            keyRequestsCounter.decrementAndGet();
            keyRequestsStorage.release(key);
        }
    }
}

