package ru.yandex.http.util.server;

import java.io.BufferedReader;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import ru.yandex.logger.FilterLogger;

public final class MemoryMonitor {
    private static final long SAMPLING_INTERVAL = 100;
    private static final CompoundLogger LOGGER = new CompoundLogger();
    private static MonitorImpl instance = null;
    private static Throwable createError = null;

    private final MonitorImpl impl;

    private MemoryMonitor(final Logger logger) {
        impl = implInstance(logger);
    }

    private MemoryMonitor() {
        impl = implInstance(null);
    }

    public static MemoryMonitor instance(final Logger logger) {
        return new MemoryMonitor(logger);
    }

    public static MemoryMonitor instance() {
        return new MemoryMonitor();
    }

    public long anonymousMemoryUsage() {
        return impl.anonymousMemoryUsage();
    }

    public long memoryLimit() {
        return impl.memoryLimit();
    }

    private static MonitorImpl implInstance(final Logger logger) {
        synchronized (LOGGER) {
            if (instance == null && createError == null) {
                try {
                    instance = createMonitor(LOGGER);
                    if (logger != null) {
                        LOGGER.addLogger(logger);
                    }
                } catch (Exception e) {
                    createError = e;
                    instance = new FakeMemoryMonitor();
                }
            }
            if (createError != null) {
                if (logger != null) {
                    logger.log(
                        Level.SEVERE,
                        "MemoryMonitor instantiate error",
                        createError);
                } else {
                    createError.printStackTrace();
                }
            }
        }
        return instance;
    }

    private static MonitorImpl createMonitor(final Logger logger)
        throws Exception
    {
        MonitorImpl impl;
        String porto = System.getProperty("PORTO");
        if ("true".equalsIgnoreCase(porto)
            || "1".equals(porto))
        {
            impl = new PortoMemoryPoller(logger, SAMPLING_INTERVAL);
        } else {
            impl = new FakeMemoryMonitor();
        }
        return impl;
    }

    private static class FakeMemoryMonitor implements MonitorImpl {
        @Override
        public long anonymousMemoryUsage() {
            return -1;
        }

        @Override
        public long memoryLimit() {
            return -1;
        }
    }

    private static class PortoMemoryPoller
        extends Thread
        implements MonitorImpl
    {
        private static final String PORTO_CTL = "portoctl";
        private static final String ANON_USAGE = "anon_usage";
        private static final String MEMORY_LIMIT_TOTAL = "memory_limit_total";
        private final Logger logger;
        private final long samplingInterval;
        private final long limit;
        private volatile long anonUsage = 0;

        PortoMemoryPoller(
            final Logger logger,
            final long samplingInterval)
            throws Exception
        {
            this.logger = logger;
            this.samplingInterval = samplingInterval;
            this.limit = portoCtlGetLong(MEMORY_LIMIT_TOTAL);
            setDaemon(true);
            start();
        }

        @Override
        public long anonymousMemoryUsage() {
            return anonUsage;
        }

        @Override
        public long memoryLimit() {
            return limit;
        }

        @Override
        public void run() {
            while (true) {
                try {
                    anonUsage = portoCtlGetLong(ANON_USAGE);
                    Thread.sleep(samplingInterval);
                } catch (Throwable t) {
                    logger.log(Level.SEVERE, "ProtoMemoryStater error", t);
                }
            }
        }

        public long portoCtlGetLong(final String propperty) throws Exception {
            Process process =
                new ProcessBuilder(PORTO_CTL, "get", "self", propperty).start();
            try (InputStream is = process.getInputStream();
                BufferedReader br =
                    new BufferedReader(
                        new InputStreamReader(
                            is,
                            StandardCharsets.UTF_8)))
            {
                String line = br.readLine();
                if (line == null) {
                    return 0;
                }
                line = line.trim();
                return Long.parseLong(line);
            } finally {
                process.waitFor();
            }
        }
    }

    private interface MonitorImpl {
        long anonymousMemoryUsage();

        long memoryLimit();
    }

    private static class CompoundLogger extends FilterLogger {
        private final CopyOnWriteArrayList<Logger> loggers;

        CompoundLogger() {
            super(Logger.getAnonymousLogger());
            loggers = new CopyOnWriteArrayList<>();
        }

        public void addLogger(final Logger logger) {
            loggers.add(logger);
        }

        @Override
        public void log(final LogRecord record) {
            for (Logger logger: loggers) {
                logger.log(record);
            }
        }
    }
}
