package ru.yandex.ljinx;

import java.util.EnumMap;
import java.util.Map;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.server.ImmutableBaseServerConfig;
import ru.yandex.stater.GolovanChart;
import ru.yandex.stater.GolovanChartGroup;
import ru.yandex.stater.GolovanPanel;
import ru.yandex.stater.GolovanSignal;
import ru.yandex.stater.ImmutableGolovanPanelConfig;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public abstract class AbstractCacheStorage implements CacheStorage {
    protected final String storageName;
    private final TimeFrameQueue<CacheResponse.CacheType> cacheStats;
    private final TimeFrameQueue<Long> siblingsRemovals;

    protected AbstractCacheStorage(
        final String storageName,
        final ImmutableBaseServerConfig serverConfig) {
        this.storageName = storageName;
        cacheStats = new TimeFrameQueue<>(serverConfig.metricsTimeFrame());
        siblingsRemovals =
            new TimeFrameQueue<>(serverConfig.metricsTimeFrame());
    }

    @Override
    public TimeFrameQueue<CacheResponse.CacheType> cacheStats() {
        return cacheStats;
    }

    @Override
    public TimeFrameQueue<Long> siblingsRemovals() {
        return siblingsRemovals;
    }


    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E {
        Map<CacheResponse.CacheType, int[]> map =
            new EnumMap<>(CacheResponse.CacheType.class);
        for (CacheResponse.CacheType type : CacheResponse.CacheType.values()) {
            map.put(type, new int[1]);
        }
        for (CacheResponse.CacheType type : cacheStats) {
            ++map.get(type)[0];
        }
        int total = 0;
        for (Map.Entry<CacheResponse.CacheType, int[]> entry : map.entrySet()) {
            int count = entry.getValue()[0];
            total += count;
            statsConsumer.stat(
                StringUtils.concat(
                    storageName,
                    '-',
                    entry.getKey().signalName(),
                    "_ammm"),
                count);
        }

        long totalSiblingsShuffles = 0L;
        long siblingsRemovals = 0L;
        for (Long stat : this.siblingsRemovals) {
            ++totalSiblingsShuffles;
            siblingsRemovals += stat.longValue();
        }

        statsConsumer.stat(
            StringUtils.concat(storageName, "-total-responses_ammm"),
            total);
        statsConsumer.stat(
            StringUtils.concat(storageName, "-total-siblings-shuffles_ammm"),
            totalSiblingsShuffles);
        statsConsumer.stat(
            StringUtils.concat(storageName, "-siblings-removals_ammm"),
            siblingsRemovals);
    }

    @Override
    public void addToGolovanPanel(
        final GolovanPanel panel,
        final String statsPrefix) {
        String prefix = StringUtils.removeSuffix(this.storageName, '-');
        String chartsPrefix = statsPrefix + prefix;
        GolovanChartGroup group =
            new GolovanChartGroup(chartsPrefix, statsPrefix);

        // mds-cache doesnot have cachehit staters
        if (!prefix.equals("mds-cache")) {
            GolovanChart chart = new GolovanChart(
                "-cache-stats",
                " Cache stats for storage: " + prefix,
                true,
                false,
                0d);

            ImmutableGolovanPanelConfig config = panel.config();

            for (CacheResponse.CacheType type : CacheResponse.CacheType.values()) {
                if (type.toString().equals("MISS")) {
                    chart.addSignal(new GolovanSignal(
                        "perc(" + chartsPrefix + "-" + "cache-misses_ammm"
                            + "," + chartsPrefix + "-total-responses_ammm)",
                        config.tag(),
                        type.signalName(),
                        null,
                        1,
                        true)
                    );
                } else {
                    chart.addSignal(new GolovanSignal(
                        "perc(" + chartsPrefix + "-" + type.toString().toLowerCase() + "-hits_ammm"
                            + "," + chartsPrefix + "-total-responses_ammm)",
                        config.tag(),
                        type.signalName(),
                        null,
                        1,
                        true)
                    );
                }
            }
            group.addChart(chart);

            chart = new GolovanChart(
                "-cache-weighted-size",
                " Cache weighted size for storage: " + prefix,
                false,
                false,
                0d);

            chart.addSplitSignal(
                config,
                chartsPrefix + "-cache-weighted-size_ammv",
                1,
                true,
                false);
            group.addChart(chart);

            chart = new GolovanChart(
                "-cache-capacity",
                " Cache capacity for storage: " + prefix,
                false,
                false,
                0d);

            chart.addSplitSignal(
                config,
                chartsPrefix + "-cache-capacity_ammv",
                1,
                true,
                false);

            group.addChart(chart);

            panel.addCharts("cache-stats", null, group);
        }
    }
}

