package ru.yandex.ljinx;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import ru.yandex.collection.IntSet;
import ru.yandex.http.config.AbstractHttpHostConfigBuilder;
import ru.yandex.http.util.request.function.RequestFunction;
import ru.yandex.ljinx.function.LjinxFunctionContext;
import ru.yandex.ljinx.function.LjinxFunctionFactory;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.searchmap.SearchMapConfig;
import ru.yandex.parser.searchmap.SearchMapConfigBuilder;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.IntSetParser;
import ru.yandex.parser.string.NonEmptyValidator;

public abstract class AbstractProxyPassConfigBuilder
    <T extends AbstractProxyPassConfigBuilder<T>>
    extends AbstractHttpHostConfigBuilder<T>
    implements ProxyPassConfig
{
    private RequestFunction<LjinxFunctionContext> cacheKey;
    private String cacheStorage;
    private IntSet cacheCodes;
    private IntSet goodCodes;
    private Set<String> noCacheHeaders;
    private Map<Integer, Long> cacheCodesTTL;
    private List<String> passHeaders;
    private SiblingsConfigBuilder siblingsConfig;
    private SearchMapConfigBuilder searchMapConfig;
    private Pattern pattern;
    private String replacement;

    protected AbstractProxyPassConfigBuilder(final ProxyPassConfig config) {
        super(config);
        cacheKey(config.cacheKey());
        cacheStorage(config.cacheStorage());
        cacheCodes(config.cacheCodes());
        goodCodes(config.goodCodes());
        noCacheHeaders(config.noCacheHeaders());
        cacheCodesTTL(config.cacheCodesTTL());
        passHeaders(config.passHeaders());
        siblingsConfig(config.siblingsConfig());
        searchMapConfig(config.searchMapConfig());
        pattern(config.pattern());
        replacement(config.replacement());
    }

    protected AbstractProxyPassConfigBuilder(
        final IniConfig config,
        final ProxyPassConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        cacheKey = config.get(
            "cache-key",
            defaults.cacheKey(),
            LjinxFunctionFactory.INSTANCE);
        cacheStorage = config.getString(
            "cache-storage",
            defaults.cacheStorage());
        cacheCodes = config.get(
            "cache-codes",
            defaults.cacheCodes(),
            IntSetParser.INSTANCE);
        goodCodes = config.get(
            "good-codes",
            defaults.goodCodes(),
            IntSetParser.INSTANCE);
        noCacheHeaders = config.get(
            "no-cache-headers",
            defaults.noCacheHeaders(),
            new CollectionParser<>(NonEmptyValidator.TRIMMED, HashSet::new));
        cacheCodesTTL = loadCacheCodesTTL(config.section("cache-codes-ttl"));
        passHeaders = config.get(
            "pass-headers",
            defaults.passHeaders(),
            new CollectionParser<>(NonEmptyValidator.TRIMMED, ArrayList::new));
        IniConfig siblings = config.sectionOrNull("siblings");
        SiblingsConfig siblingsConfigDefaults = defaults.siblingsConfig();
        if (siblings == null) {
            siblingsConfig(siblingsConfigDefaults);
        } else if (siblingsConfigDefaults == null) {
            siblingsConfig = new SiblingsConfigBuilder(siblings);
        } else {
            siblingsConfig =
                new SiblingsConfigBuilder(siblings, siblingsConfigDefaults);
        }
        IniConfig searchMap = config.sectionOrNull("searchmap");
        SearchMapConfig searchMapConfigDefaults = defaults.searchMapConfig();
        if (searchMap == null) {
            searchMapConfig(searchMapConfigDefaults);
        } else if (searchMapConfigDefaults == null) {
            searchMapConfig = new SearchMapConfigBuilder(searchMap);
        } else {
            searchMapConfig =
                new SearchMapConfigBuilder(searchMap, searchMapConfigDefaults);
        }
        String patternStr = config.getString("pattern", null);
        if (patternStr != null) {
            pattern = Pattern.compile(patternStr);
        } else {
            pattern = defaults.pattern();
        }
        replacement = config.getString(
            "replacement",
            defaults.replacement()
        );
    }

    private Map<Integer, Long> loadCacheCodesTTL(final IniConfig config)
        throws ConfigException
    {
        Map<Integer, Long> cacheCodesTTL = new HashMap<>();
        for (Map.Entry<String, IniConfig> entry
                : config.sections().entrySet())
        {
            int code = Integer.parseInt(entry.getKey());
            long ttl = entry.getValue().get(
                "ttl",
                DurationParser.POSITIVE_LONG);
            cacheCodesTTL.put(code, ttl);
        }
        return cacheCodesTTL;
    }

    @Override
    public Map<Integer, Long> cacheCodesTTL() {
        return cacheCodesTTL;
    }

    public T cacheCodesTTL(final Map<Integer, Long> cacheCodesTTL) {
        this.cacheCodesTTL = cacheCodesTTL;
        return self();
    }

    public T cacheCodeTTL(final int code, final long ttl) {
        if (cacheCodesTTL.isEmpty()) {
            cacheCodesTTL = new HashMap<>();
        }
        cacheCodesTTL.put(code, ttl);
        return self();
    }

    @Override
    public RequestFunction<LjinxFunctionContext> cacheKey() {
        return cacheKey;
    }

    public T cacheKey(final RequestFunction<LjinxFunctionContext> cacheKey) {
        this.cacheKey = cacheKey;
        return self();
    }

    @Override
    public String cacheStorage() {
        return cacheStorage;
    }

    public T cacheStorage(final String cacheStorage) {
        this.cacheStorage = cacheStorage;
        return self();
    }

    @Override
    public IntSet cacheCodes() {
        return cacheCodes;
    }

    public T cacheCodes(final IntSet cacheCodes) {
        this.cacheCodes = cacheCodes;
        return self();
    }

    @Override
    public IntSet goodCodes() {
        return goodCodes;
    }

    public T goodCodes(final IntSet goodCodes) {
        this.goodCodes = goodCodes;
        return self();
    }

    @Override
    public Set<String> noCacheHeaders() {
        return noCacheHeaders;
    }

    public T noCacheHeaders(final Set<String> noCacheHeaders) {
        this.noCacheHeaders = noCacheHeaders;
        return self();
    }

    @Override
    public List<String> passHeaders() {
        return passHeaders;
    }

    public T passHeaders(final List<String> passHeaders) {
        this.passHeaders = passHeaders;
        return self();
    }

    @Override
    public SiblingsConfigBuilder siblingsConfig() {
        return siblingsConfig;
    }

    public T siblingsConfig(final SiblingsConfig siblingsConfig) {
        if (siblingsConfig == null) {
            this.siblingsConfig = null;
        } else {
            this.siblingsConfig = new SiblingsConfigBuilder(siblingsConfig);
        }
        return self();
    }

    @Override
    public SearchMapConfigBuilder searchMapConfig() {
        return searchMapConfig;
    }

    public T searchMapConfig(final SearchMapConfig searchMapConfig) {
        if (searchMapConfig == null) {
            this.searchMapConfig = null;
        } else {
            this.searchMapConfig = new SearchMapConfigBuilder(searchMapConfig);
        }
        return self();
    }

    @Override
    public Pattern pattern() {
        return pattern;
    }

    public T pattern(final Pattern pattern) {
        this.pattern = pattern;
        return self();
    }

    @Override
    public String replacement() {
        return replacement;
    }

    public T replacement(final String replacement) {
        this.replacement = replacement;
        return self();
    }
}
